package ai.digital.config.server.api

import ai.digital.config.ConfigFileFilter
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.cloud.config.environment.Environment
import org.springframework.cloud.config.server.environment.InvalidEnvironmentRequestException
import org.springframework.cloud.config.server.support.PathUtils
import org.springframework.http.{HttpStatus, MediaType}
import org.springframework.web.bind.annotation.{PathVariable, PutMapping, RequestBody, RequestMapping, RestController}
import org.springframework.web.server.ResponseStatusException
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

@RestController
@RequestMapping(path = Array("${spring.cloud.config.server.prefix:}/"+ENCRYPT_PATH))
class CentralConfigEncryptorController(@Qualifier("environmentConfigurationPropertiesService")
                                       val centralConfigurationPropertiesService: ConfigurationPropertiesService)  extends Logging {

  @PutMapping(path = Array(PathConstants.PROPERTIES_PATH),
    consumes = Array(MediaType.APPLICATION_JSON_VALUE),
    produces = Array(MediaType.APPLICATION_JSON_VALUE))
  def propertiesEncryptDefaultLabel(@PathVariable name: String,
                             @PathVariable profiles: String,
                             @RequestBody requestBody: PropertiesEncryptRequestBody): PropertiesEncryptResponse = {
    propertiesEncrypt(name, profiles, null, requestBody)
  }

  @PutMapping(path = Array(PathConstants.PROPERTIES_PATH_WITH_LABEL),
    consumes = Array(MediaType.APPLICATION_JSON_VALUE),
    produces = Array(MediaType.APPLICATION_JSON_VALUE))
  def propertiesEncrypt(@PathVariable name: String,
                        @PathVariable profiles: String,
                        @PathVariable label: String,
                        @RequestBody requestBody: PropertiesEncryptRequestBody): PropertiesEncryptResponse = {
    try {
      normalize(name)
      normalize(profiles)
      normalize(label)
      PropertiesEncryptResponse(
        centralConfigurationPropertiesService.propertiesEncrypt(
          ConfigFileFilter(name, profiles, label),
          FileExtension.toValue(requestBody.fileExtension),
          requestBody.propertyKey,
          FilterCondition.toValue(requestBody.conditionType)
        ).asJava
      )
    } catch {
      case e: InvalidParameterException =>
        logger.error(e.getMessage, e)
        throw new ResponseStatusException(HttpStatus.BAD_REQUEST, e.getMessage)
    }
  }

  private def normalize(part: String): String = {
    if (PathUtils.isInvalidEncodedLocation(part)) throw new InvalidEnvironmentRequestException("Invalid request")
    Environment.normalize(part)
  }
}
