package ai.digital.config.server.api

import ai.digital.config.server.api.FileExtension.FileExtension
import ai.digital.config.server.api.FilterCondition.{CONTAINS, EQUALS, FilterCondition}
import ai.digital.config.server.encryptor.CentralConfigTextEncryptor
import ai.digital.config.server.{ConfigFileMappingResolver, PropertiesManagement, PropertiesManagementWithComments, YamlManagement}
import ai.digital.config.{ConfigFileFilter, ConfigFileMapping, ContainsPropertyKeyFilter, EqualPropertyKeyFilter, PropertyKeyFilter}
import grizzled.slf4j.Logging
import org.springframework.cloud.config.environment.{Environment, PropertySource}
import org.springframework.cloud.config.server.environment.{EnvironmentRepository, SearchPathLocator}
import org.springframework.context.{ApplicationContext, ApplicationContextAware}
import org.springframework.core.io.{AbstractResource, Resource, ResourceLoader, WritableResource}
import org.springframework.stereotype.Service
import org.yaml.snakeyaml.Yaml

import java.util
import java.util.{Map => JMap}
import scala.jdk.CollectionConverters._

object EnvironmentConfigurationPropertiesService {


  object FileType extends Enumeration with PropertiesManagement {

    import MapNestingFormatter.prettierFromFlatMap

    abstract class FileType(name: String, val suffixes: Set[String]) extends Val(name) {

      def replaceResource(resource: Resource,
                          header: String,
                          properties: Iterable[(String, PropertyValue)]): Unit

      def writeToResource(resource: Resource,
                          originalProperties: Map[String, AnyRef],
                          updateProperties: Map[String, PropertyValue],
                          deleteProperties: Iterable[String]): Unit

      def resolve(resource: Resource): Boolean = suffixes.exists(resource.getFilename.endsWith(_))
    }

    val Yaml: FileType = new FileType("yaml", Set(".yaml", ".yml")) with YamlManagement {

      override def replaceResource(resource: Resource, header: String, properties: Iterable[(String, PropertyValue)]): Unit = {
        implicit lazy val yamlParser: Yaml = createYamlParser()
        writeYaml(resource, prettierFromFlatMap(properties.map { case (key, value) => (key, value.value) }.toMap))
      }

      override def writeToResource(resource: Resource,
                                   originalProperties: Map[String, AnyRef],
                                   properties: Map[String, PropertyValue],
                                   deleteProperties: Iterable[String]): Unit = {
        implicit lazy val yamlParser: Yaml = createYamlParser()
        val allProperties = originalProperties ++ properties.view.mapValues(_.value)
        writeYaml(resource, prettierFromFlatMap(allProperties -- deleteProperties))
      }
    }

    val Properties: FileType = new FileType("properties", Set(".properties")) with PropertiesManagementWithComments {

      override def replaceResource(resource: Resource, header: String, properties: Iterable[(String, PropertyValue)]): Unit = {
        createResource(resource)
        replaceProperties(resource, header, properties)
      }

      override def writeToResource(resource: Resource,
                                   originalProperties: Map[String, AnyRef],
                                   properties: Map[String, PropertyValue],
                                   deleteProperties: Iterable[String]): Unit = {
        createResource(resource)
        writeProperties(resource, originalProperties, properties, deleteProperties)
      }
    }

    val Other: FileType = new FileType("other", Set.empty) {

      override def replaceResource(resource: Resource, header: String, properties: Iterable[(String, PropertyValue)]): Unit =
        throw new IllegalArgumentException(s"File type is not supported for resource $resource")

      override def writeToResource(resource: Resource,
                                   originalProperties: Map[String, AnyRef],
                                   properties: Map[String, PropertyValue],
                                   deleteProperties: Iterable[String]): Unit =
        throw new IllegalArgumentException(s"File type is not supported for resource $resource")

      override def resolve(resource: Resource): Boolean = true
    }

    val allTypes: Array[FileType] = values.toArray.map(_.asInstanceOf[FileType])

    def createResource(resource: Resource): Unit = {
      val file = resource.getFile
      if (!file.exists())
        file.createNewFile()
    }

    def replaceResource(resource: Resource, header: String, properties: Iterable[(String, PropertyValue)]): Unit =
      allTypes
        .find(_.resolve(resource))
        .foreach(_.replaceResource(resource, header, properties))

    def writeToResource(resource: Resource,
                        originalProperties: Map[String, AnyRef],
                        updateProperties: Map[String, PropertyValue],
                        deleteProperties: Iterable[String]): Unit =
      allTypes
        .find(_.resolve(resource))
        .foreach(_.writeToResource(resource, originalProperties, updateProperties, deleteProperties))
  }

  case class ResolvedResourceKey(propertySource: PropertySource, resource: Option[Resource])

  case class ResolvedResource(key: String, value: PropertyValue, resolvedResourceKey: Option[ResolvedResourceKey])

  protected object MapNestingFormatter {

    object Result {
      def apply(): FirstResult = new FirstResult

      def applyMap(): JMapResult = new JMapResult()

      def applyList(lastIndex: String): JListResult = new JListResult(lastIndex)
    }

    sealed trait Result {
      def addResult(keyPart: String, other: Result): Result

      def mergeResult(other: Result): Result

      def addValue(keyPart: String, value: AnyRef, noCheck: Boolean = false): Result
    }

    class FirstResult extends util.TreeMap[String, AnyRef] with Result {
      override def addResult(keyPart: String, other: Result): Result =
        Option(get(keyPart)) match {
          case Some(oldValue: Result) =>
            oldValue.mergeResult(other)
          case Some(oldValue: AnyRef) if other.isInstanceOf[util.Map[String, AnyRef]] =>
            other.asInstanceOf[util.Map[String, AnyRef]].put("", oldValue)
            other
          case _ =>
            put(keyPart, other)
            other
        }

      override def addValue(keyPart: String, value: AnyRef, noCheck: Boolean = false): Result = {
        val oldValue = put(keyPart, value)
        if (!noCheck && oldValue != null) {
          throw new IllegalArgumentException(s"Cannot format yaml file: overwriting old value $oldValue at position $keyPart")
        }
        this
      }

      override def mergeResult(other: Result): Result = {
        other match {
          case map: JMapResult =>
            this.putAll(map)
            this
          case _ => throw new IllegalArgumentException(s"Cannot merge map result $this with $other")
        }
      }
    }

    class JMapResult extends FirstResult

    class JListResult(var lastIndex: String) extends util.ArrayList[AnyRef] with Result {
      val propToListIndex = new util.HashMap[String, Int]()

      override def addResult(keyPart: String, other: Result): Result = {
        Option(propToListIndex.get(lastIndex)) match {
          case Some(listIndex) =>
            val oldValue = get(listIndex).asInstanceOf[Result]
            oldValue.addResult(keyPart, other)
          case _ =>
            propToListIndex.put(lastIndex, size())
            val newMap = Result.applyMap()
            newMap.addResult(keyPart, other)
            add(newMap)
            other
        }
      }

      override def addValue(keyPart: String, value: AnyRef, noCheck: Boolean = false): Result = {
        Option(propToListIndex.get(lastIndex)) match {
          case Some(listIndex) =>
            val lastResult = get(listIndex).asInstanceOf[Result]
            lastResult.addValue(keyPart, value)
            lastResult
          case _ =>
            val map = Result.applyMap()
            map.addValue(keyPart, value)
            propToListIndex.put(lastIndex, size())
            add(map)
            map
        }
      }

      def addFinalValue(keyPart: String, value: AnyRef): Result = {
        Option(propToListIndex.get(keyPart)) match {
          case Some(listIndex) =>
            set(listIndex, value)
          case _ =>
            propToListIndex.put(keyPart, size())
            add(value)
        }
        this
      }

      override def mergeResult(other: Result): Result = {
        other match {
          case list: JListResult =>
            this.lastIndex = list.lastIndex
            this.addAll(list)
            this
          case _ => throw new IllegalArgumentException(s"Cannot merge list result $this with $other")
        }
      }
    }

    private val ArrayNumber = "(.+)\\[([0-9]+)\\]$".r

    private def extractArrayNumber(str: String): Option[(String, String)] = str match {
      case ArrayNumber(keyPrefix, pos) =>
        Some((keyPrefix, pos))
      case _ => None
    }

    private def composeSlices(key: String): Seq[String] = {
      val slicedKey = {
        val slicedKey = key.split('.')
        if (slicedKey.nonEmpty) {
          slicedKey
        } else {
          Array(key)
        }
      }

      // merge "" surrounded keys
      slicedKey.foldLeft((false, Seq.empty[String])) {
        case ((false, result), keyPart) if keyPart.startsWith("\"") =>
          (true, result)
        case ((true, result), keyPart) if keyPart.endsWith("\"") =>
          (false, result)
        case ((inString, result), keyPart) => (inString, result :+ keyPart)
      }._2
    }

    private def mergeSlices(slicedKey: Seq[String], value: AnyRef, result: Result) =
      slicedKey
        .zipWithIndex
        .foldLeft((result, false)) { case ((prev, skip), (keyPart, index)) =>
          if (skip) {
            (prev, true)
          } else {
            extractArrayNumber(keyPart) match {
              case Some((keyPrefix, listIndex)) =>
                if (slicedKey.size - 1 == index) {
                  val listValue = Result.applyList(listIndex)
                  listValue.addFinalValue(listIndex, value)
                  (prev.addResult(keyPrefix, listValue), false)
                } else {
                  (prev.addResult(keyPrefix, Result.applyList(listIndex)), false)
                }
              case _ =>
                if (slicedKey.size - 1 == index) {
                  (prev.addValue(keyPart, value), false)
                } else {
                  val result = prev.addResult(keyPart, Result.applyMap())
                  result match {
                    case map: JMapResult if map.containsKey("") =>
                      prev.addValue(keyPart, map.get(""), noCheck = true)
                      prev.addValue(slicedKey.drop(index).mkString("."), value)
                      (prev, true)
                    case _ => (result, false)
                  }
                }
            }
          }
        }._1

    /**
     * This function works with flat map as input. Keys needs to be in format: "key1.subkey2.subkey3",
     * and value needs to be simple type (string, number, boolean), not list or map
     *
     * @param flatMap
     * @return formatted map
     */
    def prettierFromFlatMap(flatMap: Map[String, AnyRef]): JMap[String, AnyRef] =
      flatMap
        .toSeq
        .sortBy(_._1)
        .foldLeft(Result()) { case (result, (key, value)) =>
          try {
            val slicedKey = composeSlices(key)
            mergeSlices(slicedKey, value, result)
            result
          } catch {
            case e: RuntimeException => throw new IllegalArgumentException(s"Cannot format yaml configuration file at key $key", e)
          }
        }
  }
}

@Service
class EnvironmentConfigurationPropertiesService(configFileMappingResolver: ConfigFileMappingResolver,
                                                environmentRepository: EnvironmentRepository,
                                                searchPathLocator: SearchPathLocator,
                                                resourceLoader: ResourceLoader,
                                                textEncryptor: CentralConfigTextEncryptor)
  extends ConfigurationPropertiesService with ApplicationContextAware with Logging {

  import ai.digital.config.server.api.EnvironmentConfigurationPropertiesService._

  private var applicationContext: ApplicationContext = _

  override def propertiesStore(configFileFilter: ConfigFileFilter,
                               header: String,
                               properties: Iterable[(String, PropertyValue)],
                               configFileMapping: ConfigFileMapping): Iterable[String] =

    try {
      logger.debug(s"received properties to store with $properties for ${configFileMapping.defaultConfigFile}")
      val sourceEnvironment = getEnvironment(configFileFilter)
      val resolvedResources = resolveResources(sourceEnvironment, properties.toMap, configFileMapping)

      resolvedResources
        .flatMap {
          case (Some(ResolvedResourceKey(_, Some(resource))), resolvedResources) =>
            resourcePropertiesStore(resource, header, resolvedResources)
          case (None, resolvedResources) =>
            resolveMissingResource(configFileFilter, configFileMapping, resolvedResources)
              .flatMap {
                case (resource, resolvedResources) => resourcePropertiesStore(resource, header, resolvedResources)
              }
        }
    } catch {
      case e: RuntimeException =>
        logger.error(s"Cannot handle request with properties $properties", e)
        throw e
    }

  private def resourcePropertiesStore(resource: Resource,
                                      header: String,
                                      resolvedResources: Iterable[ResolvedResource]): Iterable[String] = {
    val storeProperties = resolvedResources
      .map(resolvedResource => resolvedResource.key -> resolvedResource.value)
      .toSeq
      .sortBy(_._1)

    FileType.replaceResource(resource, header, storeProperties)
    storeProperties.map(_._1)
  }

  override def propertiesUpdate(configFileFilter: ConfigFileFilter,
                                properties: Map[String, PropertyValue],
                                configFileMapping: ConfigFileMapping): Iterable[String] =
    try {
      logger.debug(s"received properties to update with $properties for ${configFileMapping.defaultConfigFile}")
      val sourceEnvironment = getEnvironment(configFileFilter)
      val resolvedResources = resolveResources(sourceEnvironment, properties, configFileMapping)

      resolvedResources
        .flatMap {
          case (Some(ResolvedResourceKey(propertySource, Some(resource))), resolvedResources) =>
            val originalProperties = propertySource.getSource.asScala.toMap.asInstanceOf[Map[String, PropertyValue]]
            resourcePropertiesUpdate(resource, originalProperties, resolvedResources)

          case (None, resolvedResources) =>
            resolveMissingResource(configFileFilter, configFileMapping, resolvedResources).flatMap {
              case (resource, resolvedResources) => resourcePropertiesUpdate(resource, Map.empty, resolvedResources)
            }
        }
    } catch {
      case e: RuntimeException =>
        logger.error(s"Cannot handle request with properties $properties", e)
        throw e
    }

  private def resourcePropertiesUpdate(resource: Resource,
                                       originalProperties: Map[String, PropertyValue],
                                       resolvedResources: Iterable[ResolvedResource]): Iterable[String] = {
    val updateProperties = resolvedResources
      .map(resolvedResource => resolvedResource.key -> resolvedResource.value)
      .toMap

    FileType.writeToResource(resource, originalProperties, updateProperties, Set.empty)
    updateProperties.keys
  }

  private def resolveMissingResource(configFileFilter: ConfigFileFilter,
                                     configFileMapping: ConfigFileMapping,
                                     resolvedResources: Iterable[ResolvedResource]): Iterable[(AbstractResource, Iterable[ResolvedResource])] = {
    val searchLocations = searchPathLocator.getLocations(configFileFilter.application, configFileFilter.profile, configFileFilter.label).getLocations
    val resources = filesToWritableResources(searchLocations)

    resolvedResources.map(resource => (resource, propertyKeyToFiles(resource.key, configFileMapping)))
      .filter(_._2.nonEmpty)
      .flatMap(propertyWithFiles =>
        resolveResource(resources, propertyWithFiles._2)
          .map(_ -> propertyWithFiles._1)
      )
      .groupMap(_._1)(_._2)
  }

  override def propertiesDelete(configFileFilter: ConfigFileFilter,
                                propertyKeys: Set[String],
                                configFileMapping: ConfigFileMapping): Iterable[String] =
    try {
      logger.debug(s"received properties to delete with $propertyKeys for ${configFileMapping.defaultConfigFile}")
      val sourceEnvironment = getEnvironment(configFileFilter)
      val resolvedResources = resolveResources(sourceEnvironment, propertyKeys.map(_ -> PropertyValue.empty).toMap, configFileMapping)

      resolvedResources
        .flatMap {
          case (Some(ResolvedResourceKey(propertySource, Some(resource))), resolvedResources) =>

            val originalProperties = propertySource.getSource.asScala.toMap.asInstanceOf[Map[String, AnyRef]]

            val deleteProperties = resolvedResources
              .map(resolvedResource => resolvedResource.key)

            FileType.writeToResource(resource, originalProperties, Map.empty, deleteProperties)
            deleteProperties
          case _ => Iterable.empty
        }

    } catch {
      case e: RuntimeException =>
        logger.error(s"Cannot handle request with propertyKeys $propertyKeys", e)
        throw e
    }

  private def resolveResources(environment: Option[Environment],
                               properties: Map[String, PropertyValue],
                               configFileMapping: ConfigFileMapping): Map[Option[ResolvedResourceKey], Iterable[ResolvedResource]] = {
    val sources: Iterable[PropertySource] = environment
      .toSeq
      .flatMap(_.getPropertySources.asScala).filter(s => s.getName.startsWith("file:"))

    properties
      .map { case (key, value) =>
        val resourceSource = sources
          .map(source => source -> propertyKeyToResource(key, source, configFileMapping))
          .find(_._2.isDefined)
          .map(resourceData => ResolvedResourceKey(resourceData._1, resourceData._2))
        ResolvedResource(key, value, resourceSource)
      }
      .groupBy(_.resolvedResourceKey)
  }

  private def getEnvironment(configFileFilter: ConfigFileFilter): Option[Environment] = {
    val environment = environmentRepository.findOne(configFileFilter.application, configFileFilter.profile, configFileFilter.label, false)
    if (environment == null || environment.getPropertySources.isEmpty)
      None
    else
      Some(environment)
  }

  private def propertyKeyToResource(key: String, propertySource: PropertySource, configFileMapping: ConfigFileMapping): Option[Resource] = {
    val resource = applicationContext.getResource(propertySource.getName)

    if (configFileMapping.isDefined) {
      configFileMappingResolver.propertyKeyToConfigFile(key, configFileMapping)
        .find(resource.getFilename.endsWith)
        .map(_ => resource)
    } else if (propertySource.getSource.containsKey(key)) {
      Option(resource)
    } else {
      None
    }
  }

  private def propertyKeyToFiles(key: String, configFileMapping: ConfigFileMapping): Iterable[String] = {
    if (configFileMapping.isDefined) {
      configFileMappingResolver.propertyKeyToConfigFile(key, configFileMapping)
    } else {
      Iterable.empty
    }
  }

  private def filesToWritableResources(locations: Iterable[String]): Iterable[WritableResource] =
    locations
      .map(resourceLoader.getResource)
      .filter(_.isInstanceOf[WritableResource])
      .map(_.asInstanceOf[WritableResource])

  private def resolveResource(resourceDirs: Iterable[Resource], filenames: Iterable[String]): Option[AbstractResource] =
    resourceDirs
      .flatMap(resourceDir => filenames.map(resourceDir.createRelative))
      .find(_.isReadable)
      .orElse(
        resourceDirs
          .find(_ => true)
          .flatMap(resourceDir => filenames.find(_ => true).map(resourceDir.createRelative))
      )
      .filter(_.isInstanceOf[AbstractResource])
      .map(_.asInstanceOf[AbstractResource])

  override def setApplicationContext(applicationContext: ApplicationContext): Unit = this.applicationContext = applicationContext

  override def propertiesEncrypt(configFileFilter: ConfigFileFilter, fileExtension: FileExtension,
                                 propertyKey: String, conditionType: FilterCondition): Iterable[String] = {
    try {
      logger.debug(s"received properties to encrypt with $propertyKey for $fileExtension files with $conditionType condition")

      val propertyKeyFilters: List[PropertyKeyFilter] = conditionType match {
        case CONTAINS => List(new ContainsPropertyKeyFilter(propertyKey))
        case EQUALS => List(new EqualPropertyKeyFilter(propertyKey))
        case _ => List(new ContainsPropertyKeyFilter(propertyKey))
      }

      val sourceEnvironment = getEnvironment(configFileFilter)
      val properties = sourceEnvironment.toSeq.flatMap(_.getPropertySources.asScala)
        .filter(source =>  source.getName.endsWith(fileExtension.toString.toLowerCase))
        .flatMap(source =>
          source.getSource.entrySet().asScala
            .filter( entry => propertyKeyFilters.exists(_.shouldBeEncrypted(entry.getKey.toString))  &&
              !textEncryptor.isEncrypted(entry.getValue.toString))
            .map(entry => (entry.getKey.toString, PropertyValue(
              textEncryptor.encrypt(entry.getValue.toString)))))
        .toMap

      logger.info(s"Encrypting following properties ${properties.keys.mkString(", ")}")

      propertiesUpdate(configFileFilter, properties, ConfigFileMapping())
    } catch {
      case e: RuntimeException =>
        logger.error(s"Cannot handle encrypt request with properties ", e)
        throw e
    }
  }
}
