package ai.digital.engine.exportcis.archive

import com.xebialabs.deployit.core.xml.PasswordEncryptingCiConverter
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind.{CI, SET_OF_CI}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.artifact.{Artifact, SourceArtifact}
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.repository.internal.Root
import com.xebialabs.deployit.repository.{ConfigurationItemData, RepositoryService, SearchParameters}
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.xlplatform.artifact.resolution.ArtifactResolverRegistry.isStoredArtifact
import ai.digital.engine.exportcis.archive.RepositoryExporter.RepositoryRootAlias
import ai.digital.engine.exportcis.archive.RepositoryXmlExporter.Attributes._
import ai.digital.engine.exportcis.archive.RepositoryXmlExporter.Elements._
import ai.digital.engine.exportcis.archive.RepositoryXmlExporter._
import com.xebialabs.xltype.serialization.jdom.{CiJdomWriter, JdomWriter}
import com.xebialabs.xltype.serialization.{CiWriter, ConfigurationItemConverter}
import grizzled.slf4j.Logging
import org.jdom2.Namespace
import org.joda.time.DateTime
import java.util

import scala.jdk.CollectionConverters._

object RepositoryXmlExporter {
  val XlNamespace: Namespace = Namespace.getNamespace("xl", "http://www.xebialabs.com/xl-export")

  object Elements {
    val export = "export"
    val metadata = "metadata"
    val exportedBy = "exported-by"
    val exportedAt = "exported-at"
    val exportedFormatVersion = "exported-format-version"
    val encryptionKeyFingerprint = "encryption-key-fingerprint"
    val exportedConfigurationItems = "exported-configuration-items"
  }

  object Attributes {
    val exportedRootId = "exported-root-id"
  }

  private[archive] type FilesByZipEntry = Map[String, OverthereFile]
}

case class RepositoryXmlExporterResult(xmlContent: String, filesByZipEntry: FilesByZipEntry)

class RepositoryXmlExporter(val repositoryService: RepositoryService, val ctx: ExecutionContext) extends Logging {

  def exportCiTrees(ciRoots: Iterable[String], userName: String, appVersion: String): RepositoryXmlExporterResult = {
    ctx.logOutput(s"Started exporting CIs under ${ciRoots.size} roots into an XML format")

    val writer = new JdomWriter
    writer.element(export, XlNamespace)
    writeMetadata(userName, appVersion, writer)


    val ciWriter = new RepositoryXmlCiConverter

    for (root <- ciRoots) {
      writer.element(exportedConfigurationItems, XlNamespace).attribute(exportedRootId, root)
      writeCiTree(root, writer, ciWriter)
      writer.endElement()
    }
    writer.endElement()

    ctx.logOutput(s"Finished exporting CIs under ${ciRoots.size} roots into an XML format")

    RepositoryXmlExporterResult(writer.toString, ciWriter.filesByZipEntry)
  }

  private def writeMetadata(userName: String, appVersion: String, writer: JdomWriter): Unit = {
    writer.element(metadata, XlNamespace).
      element(exportedBy, XlNamespace).value(userName).endElement().
      element(exportedAt, XlNamespace).valueAsDate(DateTime.now).endElement().
      element(exportedFormatVersion, XlNamespace).value(appVersion).endElement().
      element(encryptionKeyFingerprint, XlNamespace).value(PasswordEncrypter.getInstance.getKeyFingerprint).endElement().
      endElement()
  }

  private def writeCiTree(rootId: String, writer: JdomWriter, ciWriter: ConfigurationItemConverter): Unit = {
    ctx.logOutput(s"Searching for CIs in subtree [$rootId]")

    val allCisSearchParams = new SearchParameters()

    val cis: Seq[ConfigurationItemData] = rootId match {
      case RepositoryRootAlias =>
        repositoryService.list(allCisSearchParams).asScala.toSeq
      case _ =>
        (new ConfigurationItemData(rootId, null) +: repositoryService.list(allCisSearchParams.setAncestor(rootId)).asScala).toSeq
    }

    ctx.logOutput(s"Found ${cis.size} CIs in subtree [$rootId], fetching their metadata")

    for {
      ciData <- cis
      ci = repositoryService.read[ConfigurationItem](ciData.getId, 1)
      if !ci.isInstanceOf[Root]
    } {
      logger.trace(s"Exporting CI [${ci.getId}] into the XML file")
      ciWriter.writeCi(ci, new CiJdomWriter(writer))
    }

    ctx.logOutput(s"Finished exporting subtree [$rootId]")
  }

}

private class RepositoryXmlCiConverter extends PasswordEncryptingCiConverter {

  setWriteValidationMessages(false)

  var filesByZipEntry: FilesByZipEntry = Map()

  override def writeCi(ci: ConfigurationItem, writer: CiWriter, ignored: Int): Unit = {
    writer.startCi(ci.getType.toString, ci.getId)
    ci match {
      case bci: BaseConfigurationItem => writer.ciAttributes(bci.get$ciAttributes())
      case _ =>
    }
    ci match {
      case artifact: SourceArtifact if isStoredArtifact(artifact) =>
        val entryPath = artifactZipEntryPath(artifact)
        filesByZipEntry = filesByZipEntry.updated(entryPath, artifact.getFile)
        writer.ciFileAttribute(entryPath)
      case _ =>
    }
    writeProperties(ci, writer, 0)
    writeLookupValues(ci, writer)
    writer.endCi()
  }

  override protected def writeProperty(ci: ConfigurationItem, property: PropertyDescriptor, writer: CiWriter, ciRefsFromLevel: Int): Unit = {
    if (util.EnumSet.of(CI, SET_OF_CI).contains(property.getKind) && property.isAsContainment) {
      return
    }
    super.writeProperty(ci, property, writer, ciRefsFromLevel)
  }

  private def artifactZipEntryPath(artifact: Artifact): String = {
    s"${artifact.getId}/${artifact.getFile.getName}"
  }
}
