package com.xebialabs.deployit.deployment;

import com.google.common.base.Function;
import com.google.common.collect.*;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecificationWithDependencies;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.repository.ChangeSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;

import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.CREATE;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.DESTROY;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static java.util.EnumSet.of;

public class ChangeSetBuilder {

    public static final Function<ConfigurationItem, String> CI_2_NAME_FUNCTION = ConfigurationItem::getName;

    private static final String CI_DEBUG = "The CI [{}] will be {} in the repository";

    private ChangeSetBuilder() {}


    public static ChangeSet determineChanges(DeltaSpecificationWithDependencies specification) {
        ChangeSet changeSet = new ChangeSet();
        for (DeltaSpecification deltaSpecification : specification.getAllDeltaSpecifications()) {
            determineChanges(changeSet, deltaSpecification);
        }
        return changeSet;
    }

    public static ChangeSet determineChanges(DeltaSpecification specification) {
        ChangeSet changeSet = new ChangeSet();
        determineChanges(changeSet, specification);
        return changeSet;
    }

    private static void determineChanges(ChangeSet changeSet, DeltaSpecification specification) {
        logger.debug("Determining change set for deployed application [{}]", specification.getDeployedApplication());
        addDeployedApplication(changeSet, specification);
        DeployedApplication xDeployedApplication = specification.getDeployedApplication();
        changeSet.setForceRedeploy(xDeployedApplication.isForceRedeploy());
        for (Delta delta : specification.getDeltas()) {
            switch (delta.getOperation()) {
                case CREATE:
                    changeSet.create(delta.getDeployed());
                    addEmbeddeds(changeSet, null, delta.getDeployed(), CREATE);
                    xDeployedApplication.addDeployed(delta.getDeployed());
                    logger.debug(CI_DEBUG, delta.getDeployed().getId(), delta.getOperation());
                    break;
                case MODIFY:
                case NOOP:
                    changeSet.update(delta.getDeployed());
                    addEmbeddeds(changeSet, delta.getPrevious(), delta.getDeployed(), delta.getOperation());
                    xDeployedApplication.addDeployed(delta.getDeployed());
                    logger.debug(CI_DEBUG, delta.getDeployed().getId(), delta.getOperation());
                    break;
                case DESTROY:
                    if (!(delta.getPrevious() instanceof BaseDeployedContainer)) {
                        changeSet.delete(delta.getPrevious());
                        addEmbeddeds(changeSet, delta.getPrevious(), null, DESTROY);
                        logger.debug(CI_DEBUG, delta.getPrevious().getId(), delta.getOperation());
                    }
                    break;
            }
        }

        logger.debug("DeployedApplication [{}] will contain the following {} deployeds: {}", xDeployedApplication, xDeployedApplication.getDeployeds().size(), xDeployedApplication.getDeployeds());
    }

    private static void addDeployedApplication(ChangeSet changeSet, DeltaSpecification specification) {
        DeployedApplication deployedApplication = specification.getDeployedApplication();
        switch (specification.getOperation()) {
            case CREATE:
                changeSet.create(deployedApplication);
                break;
            case MODIFY:
                if(deployedApplication.isForceRedeploy()) {
                    //Remove env and add new env
                    changeSet.delete(deployedApplication);
                    changeSet.create(deployedApplication);
                } else {
                    changeSet.update(deployedApplication);
                }
                break;
            case DESTROY:
                changeSet.delete(deployedApplication);
                break;
            case NOOP:
                break;
        }
    }

    public static void addEmbeddeds(ChangeSet changeSet, EmbeddedDeployedContainer<?, ?> previous, EmbeddedDeployedContainer<?, ?> current, Operation operation) {
        EmbeddedDeployedContainer edc = operation != DESTROY ? current : previous;

        logger.trace("Going to scan whether {}[{}] has embeddeds", edc.getType(), edc.getId());
        for (PropertyDescriptor propertyDescriptor : edc.getType().getDescriptor().getPropertyDescriptors()) {
            logger.trace("Analyzing property [{}]", propertyDescriptor.getFqn());
            if (!isEmbeddedProperty(propertyDescriptor)) {
                logger.trace("[{}] is not an embedded property", propertyDescriptor.getFqn());
                continue;
            }

            logger.debug("Determining changeset for embeddeds in property [{}] of [{}]", propertyDescriptor.getFqn(), edc.getId());

            if (operation == CREATE || operation == DESTROY) {
                //noinspection unchecked
                Collection<EmbeddedDeployed> embeddeds = (Collection<EmbeddedDeployed>) propertyDescriptor.get(edc);
                if (embeddeds == null || embeddeds.isEmpty()) {
                    continue;
                }

                for (EmbeddedDeployed embedded : embeddeds) {
                    if (operation == CREATE) {
                        if (changeSet.isForceRedeploy()) {
                            changeSet.create(embedded);
                        } else {
                            changeSet.createOrUpdate(embedded);
                        }

                        logger.debug("The CI {} will be {} in the repository", embedded.getId(), operation);
                        addEmbeddeds(changeSet, null, embedded, operation);
                    } else {
                        // Nothing added as parent is deleted
                        logger.debug("The CI {} will be {} in the repository", embedded.getId(), operation);
                        addEmbeddeds(changeSet, embedded, null, operation);
                    }
                }
            } else {
                // We need to deal with 2 sets to do difference detection
                //noinspection unchecked
                Collection<EmbeddedDeployed> currentEmbeddeds = (Collection<EmbeddedDeployed>) propertyDescriptor.get(current);
                currentEmbeddeds = currentEmbeddeds != null ? currentEmbeddeds : Sets.<EmbeddedDeployed>newHashSet();
                //noinspection unchecked
                Collection<EmbeddedDeployed> previousEmbeddeds = (Collection<EmbeddedDeployed>) propertyDescriptor.get(previous);
                previousEmbeddeds = previousEmbeddeds != null ? previousEmbeddeds : Sets.<EmbeddedDeployed>newHashSet();

                ImmutableListMultimap<String, EmbeddedDeployed> nameToCiMap = Multimaps.index(Iterables.concat(previousEmbeddeds, currentEmbeddeds), CI_2_NAME_FUNCTION);
                for (String ciName : nameToCiMap.keySet()) {
                    ImmutableList<EmbeddedDeployed> embeddedDeployeds = nameToCiMap.get(ciName);
                    if (embeddedDeployeds.size() == 2) {
                        EmbeddedDeployed embedded = embeddedDeployeds.get(1);
                        changeSet.update(embedded);
                        logger.debug("The CI {} will be {} in the repository", embedded.getId(), operation);
                        addEmbeddeds(changeSet, embeddedDeployeds.get(0), embedded, operation);
                    } else if (embeddedDeployeds.size() == 1) {
                        // We only saw one embeddedDeployed with this name, this means it is either created or deleted, which one is it?
                        EmbeddedDeployed embedded = embeddedDeployeds.get(0);
                        if (currentEmbeddeds.contains(embedded)) {
                            changeSet.createOrUpdate(embedded);
                            logger.debug("The CI {} will be {} in the repository", embedded.getId(), CREATE);
                            addEmbeddeds(changeSet, null, embedded, CREATE);
                        } else {
                            changeSet.delete(embedded);
                            logger.debug("The CI {} will be {} in the repository", embedded.getId(), DESTROY);
                            addEmbeddeds(changeSet, embedded, null, DESTROY);
                        }
                    } else {
                        throw new IllegalStateException("Shouldn't have seen more than 2 ConfigurationItems with the same name (" + ciName + "): " + nameToCiMap);
                    }
                }
            }
        }
    }

    private static boolean isEmbeddedProperty(PropertyDescriptor propertyDescriptor) {
        return propertyDescriptor.isAsContainment()
                && of(SET_OF_CI, LIST_OF_CI).contains(propertyDescriptor.getKind())
                && propertyDescriptor.getReferencedType().isSubTypeOf(Type.valueOf(EmbeddedDeployed.class));
    }

    private static final Logger logger = LoggerFactory.getLogger(ChangeSetBuilder.class);

}
