package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.deployment.ChangeSetBuilder
import com.xebialabs.deployit.deployment.planner.DeltaSpecificationBuilder._
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation._
import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, DeltaSpecification, Operation}
import com.xebialabs.deployit.plugin.api.reflect.Descriptor
import com.xebialabs.deployit.plugin.api.udm.{Container, Deployable, Deployed, DeployedApplication}
import com.xebialabs.deployit.repository.{ChangeSet, RepositoryServiceHolder}
import org.slf4j.LoggerFactory

import java.util
import scala.collection.mutable
import scala.jdk.CollectionConverters._

class CheckpointManager(fullSpec: MultiDeltaSpecification) extends Serializable {

  import CheckpointManager._

  @transient private lazy val logger = LoggerFactory.getLogger(getClass)

  private[this] val checkpointDeltas: List[Map[String, CheckpointDelta]] = fullSpec
    .getAllGroupedDeltaSpecifications
    .asScala
    .map(_.asScala.flatMap { ds => ds.getDeltas.asScala.map(d => (d.deployed.getId, new CheckpointDelta(d, ds))) }.toMap)
    .toList

  private[this] val checkpointDeltaSpecifications: List[Map[String, CheckpointDeltaSpecification]] = fullSpec
    .getAllGroupedDeltaSpecifications
    .asScala
    .map(_.asScala.map(ds => ds.deployedApplication.getId -> new CheckpointDeltaSpecification(ds, affected = false)).toMap)
    .toList

  def markDeployedApplication(deployedApplicationId: String): Unit =
    checkpointDeltaSpecifications.find(x => x.contains(deployedApplicationId))
      .flatMap(map => map.get(deployedApplicationId))
      .foreach(checkpointDeltaSpecification => checkpointDeltaSpecification.affected = true)

  def markCheckpoint(id: String, operation: Operation, intermediateCheckpointName: String): Unit =
    checkpointDeltas
      .find(_ contains id)
      .collect { case map if map contains id => map(id) }
      .collect { case checkpointDelta =>
        checkpointDelta.markCheckpoint(operation, intermediateCheckpointName)
        markDeployedApplication(checkpointDelta.ds.deployedApplication.getId)
      }

  def prepareRollback(): MultiDeltaSpecification = {
    def determineDeltaSpecificationOperation(builder: DeltaSpecificationBuilder,
                                             operation: Operation,
                                             deployedApplication: DeployedApplication,
                                             previouslyDeployedApplication: DeployedApplication,
                                             destroyedDeployeds: Set[Deployed[_ <: Deployable, _ <: Container]]): Unit =
      operation match {
        case CREATE =>
          builder.undeploy(deployedApplication)
        case DESTROY if previouslyDeployedApplication.getDeployeds.asScala.toSet.diff(destroyedDeployeds).isEmpty =>
          logger.info("The undeployment was rolled back after completion, fully redeploying the previous deployment.")
          builder.initial(previouslyDeployedApplication)
        case DESTROY =>
          builder.upgrade(deployedApplication, previouslyDeployedApplication)
        case MODIFY if deployedApplication.getDeployeds.asScala.toSet.diff(destroyedDeployeds).isEmpty =>
          logger.info("The update deployment was rolled back when there were no deployeds left, fully redeploying the previous deployment.")
          builder.initial(previouslyDeployedApplication)
        case MODIFY =>
          builder.upgrade(deployedApplication, previouslyDeployedApplication)
        case NOOP =>
          throw new IllegalStateException("DeltaSpecification can never have a NOOP operation.")
      }
    val deployedsToCreateOnForceDeploy = mutable.Set[String]()
    val specs = checkedSpecsToDeltas(includeAll)
      .reverse
      .map(_.view
        .reverseIterator
        .map { case (spec, checkpointDeltas) =>
          val builder: DeltaSpecificationBuilder = newRollbackSpecification
          val destroyedDeployeds = mutable.Set[Deployed[_ <: Deployable, _ <: Container]]()

          checkpointDeltas.foreach { cp =>
            cp.getOperation match {
              case Operation.DESTROY if cp.fullyCheckpointed =>
                destroyedDeployeds.add(cp.deployed)
                cp.rollback().foreach(builder.`with`)
              case _ =>
                cp.rollback().foreach(builder.`with`)
            }
          }
          if(spec.getDeployedApplication.isForceRedeploy) {
            spec.getPreviousDeployedApplication.getDeployeds.forEach(deployed => {
              if(!deployedsToCreateOnForceDeploy.contains(deployed.getId)) {
                builder.create(deployed)
                deployedsToCreateOnForceDeploy.add(deployed.getId)
              }
            })
          }
          determineDeltaSpecificationOperation(builder, spec.getOperation, spec.getDeployedApplication, spec.getPreviousDeployedApplication, destroyedDeployeds.toSet)
          builder.build
        }
        .toList
        .asJava
      )
    MultiDeltaSpecification.forRollback(specs.asJava, fullSpec.getMainApplication)
  }

  private[this] def isMarkedDelta(delta: CheckpointDelta) = delta.fullyCheckpointed || delta.getIntermediateCheckpoints.asScala.nonEmpty

  private[this] def includeAll(delta: CheckpointDelta) = true

  private[this] def checkedSpecsToDeltas(filter: CheckpointDelta => Boolean): List[List[(DeltaSpecification, List[CheckpointDelta])]] = {
    val allCheckpointDeltas: Map[String, CheckpointDelta] = checkpointDeltas
      .reduceOption(_ ++ _)
      .getOrElse(Map.empty)
    markSpecsWithoutCheckpoints(allCheckpointDeltas)
    val affectedSpecs: List[List[DeltaSpecification]] = checkpointDeltaSpecifications
      .map(_.collect { case (_, cds) if cds.affected => cds.ds }.toList)
      .filter(_.nonEmpty)
    val mapped = affectedSpecs
      .map(_.map(spec => spec -> allCheckpointDeltas.values.collect { case cd if filter(cd) && cd.ds == spec => cd }.toList))
    //Return at least main application when no checkpoints are found
    if (mapped.nonEmpty) mapped else List(List(fullSpec.getAllDeltaSpecifications.get(0) -> Nil))
  }

  private[this] def markSpecsWithoutCheckpoints(allCheckpointDeltas: Map[String, CheckpointDelta]): Unit = {
    checkpointDeltaSpecifications
      .flatMap(_.values
        .filter(cds => !cds.affected && (cds.ds.getOperation == MODIFY || cds.ds.getOperation == DESTROY) && cds.ds.getDeltas.asScala.nonEmpty))
      .foreach { checkpointDeltaSpecification =>
        checkpointDeltaSpecification.affected = true
        allCheckpointDeltas
          .values
          .filter(_.ds == checkpointDeltaSpecification.ds)
          .foreach(_.fullyCheckpointed = true)
      }
  }

  def changeSet: ChangeSet = {
    val changeSet = new ChangeSet
    checkedSpecsToDeltas(isMarkedDelta).foreach(x => x.foreach { case (spec, deltas) =>
      val partialDeployedApplication = getPartialDeployedApplication(spec)
      deltas.foreach { cp =>
        cp.getOperation match {
          case CREATE =>
            partialDeployedApplication.addDeployed(cp.getDeployed)
            changeSet.createOrUpdate(cp.getDeployed)
            ChangeSetBuilder.addEmbeddeds(changeSet, cp.getPrevious, cp.getDeployed, CREATE)
          case DESTROY =>
            partialDeployedApplication.getDeployeds.remove(cp.getPrevious)
            changeSet.delete(cp.getPrevious)
            ChangeSetBuilder.addEmbeddeds(changeSet, cp.getPrevious, cp.getDeployed, DESTROY)
          case MODIFY | NOOP =>
            partialDeployedApplication.getDeployeds.remove(cp.getPrevious)
            partialDeployedApplication.addDeployed(cp.getDeployed)
            changeSet.createOrUpdate(cp.getDeployed)
            ChangeSetBuilder.addEmbeddeds(changeSet, cp.getPrevious, cp.getDeployed, CREATE)
        }
      }

      if (spec.getOperation == Operation.CREATE && partialDeployedApplication.getDeployeds.isEmpty) {
        changeSet.delete(partialDeployedApplication)
      } else {
        changeSet.createOrUpdate(partialDeployedApplication)
      }
    })
    changeSet
  }

  private[this] def duplicateDeployedApplication(deployedApplication: DeployedApplication) = {
    val descriptor: Descriptor = deployedApplication.getType.getDescriptor
    val nda: DeployedApplication = descriptor.newInstance(deployedApplication.getId)
    descriptor.getPropertyDescriptors.asScala.filterNot(_.getName == "deployeds").foreach { pd =>
      pd.set(nda, pd.get(deployedApplication))
    }
    nda.setDeployeds(new util.HashSet(deployedApplication.getDeployeds))
    nda
  }

  private[this] def getPartialDeployedApplication(spec: DeltaSpecification) = spec.getOperation match {
    case CREATE =>
      val nda = duplicateDeployedApplication(spec.getDeployedApplication)
      nda.getDeployeds.clear()
      nda
    case _ =>
      val latestDeployedApplication: DeployedApplication = RepositoryServiceHolder.getRepositoryService.read(spec.getPreviousDeployedApplication.getId, 1)
      val deployedApplication: DeployedApplication = if (latestDeployedApplication != null) latestDeployedApplication else spec.getPreviousDeployedApplication
      duplicateDeployedApplication(deployedApplication)
  }
}

private class CheckpointDeltaSpecification(val ds: DeltaSpecification, var affected: Boolean) extends Serializable

private class CheckpointDelta(d: Delta, val ds: DeltaSpecification) extends Delta {

  import CheckpointManager._

  @transient private lazy val logger = LoggerFactory.getLogger(getClass)

  var fullyCheckpointed: Boolean = false
  private val intermediateCheckpoints = mutable.ListBuffer[String]()
  private var overrideOperation: Option[Operation] = None

  override def getDeployed: Deployed[_ <: Deployable, _ <: Container] = d.getDeployed.asInstanceOf[Deployed[Deployable, Container]]

  override def getOperation: Operation = overrideOperation.getOrElse(d.getOperation)

  override def getPrevious: Deployed[_ <: Deployable, _ <: Container] = d.getPrevious.asInstanceOf[Deployed[Deployable, Container]]

  /* Do not go to super, the intermediateCheckpoints is immutable */
  override def getIntermediateCheckpoints: util.List[String] = intermediateCheckpoints.asJava

  def markCheckpoint(operation: Operation, intermediateCheckpointName: String): Unit = {
    overrideOperation = calculateOverrideOperation(operation)

    Option(intermediateCheckpointName).filterNot(_.isEmpty) match {
      case Some(x) if !fullyCheckpointed =>
        logger.info(s"Marked intermediate checkpoint $x for ${this.deployed.getId}")
        intermediateCheckpoints.addOne(x)
      case Some(x) =>
        logger.info(s"Cannot mark intermediate checkpoint $x for ${this.deployed.getId} as it is already fully checkpointed")
      case None =>
        logger.info(s"Marked checkpoint for ${this.deployed.getId}")
        intermediateCheckpoints.clear()
        fullyCheckpointed = true
    }
  }

  def calculateOverrideOperation(operation: Operation): Option[Operation] = {
    (d.getOperation, operation) match {
      case (MODIFY, DESTROY) => Some(DESTROY)
      case (MODIFY, CREATE) if overrideOperation.contains(DESTROY) => None
      case (x, y) if y != null && x != y =>
        logger.error(s"Cannot mark checkpoint with operation $y for delta ${d.deployed.getId} with operation $x")
        // Plugin provided a wrong override operation, ignore it.
        None
      case _ =>
        // Delta operation and checkpoint operation match, or checkpoint had a null operation
        None
    }
  }

  def rollback(): Option[Delta] = {
    val hasHitCheckpoint: Boolean = fullyCheckpointed || intermediateCheckpoints.nonEmpty

    import Deltas._
    getOperation match {
      case Operation.CREATE if !hasHitCheckpoint =>
        None
      // A create that has hit no checkpoint has no rollback variant
      case _ if !hasHitCheckpoint =>
        // Any other operation which has not hit a checkpoint is a NOOP in a rollback
        Some(noOp(getPrevious))
      case Operation.CREATE =>
        // A create which hit a checkpoint is a DESTROY (with possible intermediate checkpoints)
        Some(destroy(getDeployed, intermediateCheckpoints.toList))
      case Operation.DESTROY if hasHitCheckpoint =>
        // A Destroy which is hit a checkpoint is a CREATE with checkpoints
        Some(create(getPrevious, intermediateCheckpoints.toList))
      case Operation.MODIFY if hasHitCheckpoint =>
        Some(modify(getDeployed, getPrevious, intermediateCheckpoints.toList))
      case _ =>
        Some(noOp(getPrevious))
    }
  }
}


object CheckpointManager {

  implicit class DeltaUtils(val delta: Delta) extends AnyVal {
    def deployed: Deployed[_ <: Deployable, _ <: Container] = if (delta.getDeployed != null) delta.getDeployed else delta.getPrevious
  }

  implicit class DeltaSpecUtils(val deltaSpec: DeltaSpecification) extends AnyVal {
    def deployedApplication: DeployedApplication = if (deltaSpec.getDeployedApplication != null) {
      deltaSpec.getDeployedApplication
    } else {
      deltaSpec.getPreviousDeployedApplication
    }
  }

}

object Deltas {
  def destroy(d: Deployed[_ <: Deployable, _ <: Container], intermediateCheckpoints: List[String] = Nil): Delta = {
    new DefaultDelta(DESTROY, d, null, intermediateCheckpoints.asJava)
  }

  def create(d: Deployed[_ <: Deployable, _ <: Container], intermediateCheckpoints: List[String] = Nil): Delta = {
    new DefaultDelta(CREATE, null, d, intermediateCheckpoints.asJava)
  }

  def modify(previous: Deployed[_ <: Deployable, _ <: Container],
             wanted: Deployed[_ <: Deployable, _ <: Container],
             intermediateCheckpoints: List[String] = Nil): Delta = {
    new DefaultDelta(MODIFY, previous, wanted, intermediateCheckpoints.asJava)
  }

  def noOp(d: Deployed[_ <: Deployable, _ <: Container], intermediateCheckpoints: List[String] = Nil): Delta = {
    new DefaultDelta(NOOP, d, d, intermediateCheckpoints.asJava)
  }
}
