package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.deployment.planner.CheckpointManager._
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState
import com.xebialabs.deployit.engine.spi.execution.{NonRemotableExecutionStateListener, StepExecutionStateEvent, TaskExecutionStateEvent}
import com.xebialabs.deployit.engine.tasker.TaskStep
import com.xebialabs.deployit.plugin.api.flow.Step
import org.slf4j.LoggerFactory

import java.util
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._

class CheckPointManagerListener(@BeanProperty val fullSpec: MultiDeltaSpecification,
                                checkpoints: util.List[StepPlan.Checkpoint]) extends NonRemotableExecutionStateListener {
  @transient private lazy val logger = LoggerFactory.getLogger(getClass)

  @BeanProperty val checkpointManager = new CheckpointManager(fullSpec)

  private var currentTaskState: TaskExecutionState = TaskExecutionState.PENDING

  override def taskStateChanged(event: TaskExecutionStateEvent): Unit = {
    currentTaskState = event.currentState()
  }

  def getCurrentTaskState: TaskExecutionState = currentTaskState

  override def stepStateChanged(event: StepExecutionStateEvent): Unit = {
    logger.debug(s"Got event [$event]")
    val step: TaskStep = event.step().asInstanceOf[TaskStep]
    // SKIPPED or DONE steps are checkpointed.
    if (step.getState.isFinal) {
      step
        .getMetadata
        .asScala
        .view
        .filterKeys(_.startsWith("checkpointId"))
        .toList
        .sortBy(_._1)
        .foreach { case (_, cpId) => markCheckpoints((cpId, step.getImplementation)) }
    }
  }

  private[this] def markCheckpoints(tuple: (String, Step)): Unit = tuple match {
    case (_, x: RepositoryUpdateStep) =>
      // if the step is a RepositoryUpdateStep there is no checkpoint to set, so we mark the DeltaSpecification as affected.
      x.spec.getAllDeltaSpecifications.asScala.toList.foreach { spec =>
        checkpointManager.markDeployedApplication(spec.getDeployedApplication.getId)
      }
    case (null, _) => // No checkpoint is needed.
    case (checkpointId, _) =>
      checkpoints
        .asScala
        .view
        .filter(_.getId == checkpointId)
        .foreach { cp =>
          val delta = cp.getDelta
          val deployedId = delta.deployed.getId
          checkpointManager.markCheckpoint(deployedId, cp.getOperation, cp.getIntermediateCheckpointName)
        }
  }
}
