package ai.digital.deploy.notification.service

import ai.digital.deploy.notification.events.UserTokenAboutToExpireEvent
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.plugin.mail.SmtpServer
import com.xebialabs.deployit.repository.{ConfigurationRepository, RepositoryService}
import com.xebialabs.deployit.security.service.{UserProfileService, UserTokenService}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ai.digital.deploy.notification.email.MultipartEmailSender
import com.xebialabs.deployit.service.dependency.RepositoryServiceAware
import grizzled.slf4j.Logging

import java.time.temporal.ChronoUnit
import java.time.{Instant, LocalDate, ZoneId}
import java.util.Date
import scala.jdk.CollectionConverters.CollectionHasAsScala

/**
 * Service for sending email notifications on Personal Access Token expiration.
 * NOTE: This service directly accesses the ConfigurationRepository to retrieve settings.
 * Any changes to keys or structure in General Settings must be reflected in this class.
 */
@Component
class PATEmailNotificationService @Autowired()(userProfileService: UserProfileService,
                                               userTokenService: UserTokenService,
                                               multipartEmailSender: MultipartEmailSender,
                                               configurationRepository: ConfigurationRepository,
                                               override val repositoryService: RepositoryService) extends Logging  with RepositoryServiceAware {

  private lazy val serverUrl: String = ServerConfiguration.getInstance().getServerUrl

  def sendNotification(event: UserTokenAboutToExpireEvent): Unit = {
    logger.info(s"Sending email notification for expiring PAT token for user: ${event.username}")

    val settings = getGeneralSettings()
    val mailServerRefOpt = settings.get("pat_email_notification_smtp_server_ci_ref")
    val logoFileOpt = settings.get("logo_file")

    if (mailServerRefOpt.isEmpty) {
      logger.info("PAT Notification: Email server is not set properly in general settings. Hence skipping PAT notification.")
      return
    }

    val mailServerOpt = mailServerRefOpt.flatMap(getMailServer(_, event.userToken.getCiId))
    val emailOpt = getUserEmail(event.username)

    for {
      mailServer <- mailServerOpt
      email <- emailOpt
    } sendExpiryEmail(mailServer, email, event, logoFileOpt)
  }

  private def sendExpiryEmail(
                               mailServer: SmtpServer,
                               email: String,
                               event: UserTokenAboutToExpireEvent,
                               logoFileOpt: Option[String]): Unit = {
    val expiryDate = event.userToken.getExpiryDate
    val hasCustomLogo = logoFileOpt.exists(_.nonEmpty)
    val customLogoParam = logoFileOpt.filter(_.nonEmpty).map(logo => ("customLogo", logo))

    multipartEmailSender.sendCustomEmail(
      mailServer = mailServer, // SMTP server to use
      from = mailServer.getFromAddress, // Sender address
      to = List(email), // Recipient(s)
      subject = "[Deploy] Your personal access token is about to expire", // Subject
      templateVariables = Map(
        "tokenNote" -> event.userToken.getTokenNote,
        "tokenExpiryDate" -> formatExpiryDate(expiryDate),
        "accessTokensUrl" -> s"$serverUrl#/personalsettings/tokens",
        "url" -> serverUrl,
        "isCustomLogoSettingsDefined" -> hasCustomLogo
      ), // Variables for template
      baseTemplatePath = Some("templates/email_base_template.ftl"), // Required base template
      contentTemplate = Some("email_content_access_token.ftl"), // content template
      textContent = None, // No plain text content
      inlineImages = Seq("deploy-logo-white" -> "images/deploy-logo.png"), // Inline images
      customLogo = customLogoParam // custom logo
    )
  }

  private def getUserEmail(username: String): Option[String] = {
    try {
      val userEmailOpt = Option(userProfileService.findOne(username)).flatMap { profile =>
        Option(profile.email)
      }
      userEmailOpt match {
        case Some(email) if email.trim.nonEmpty => Some(email)
        case _ =>
          logger.info("User email not configured. Hence skipping PAT notification.")
          None
      }
    } catch {
      case _ : com.xebialabs.deployit.exception.NotFoundException =>
        logger.info(s"No user profile found for user: $username. Hence skipping PAT notification.")
        None
    }
  }

  //TODO: Add token id as an input param and uncomment updateTokenExpiredNotified
  private def getMailServer(smtpId: String, tokenId: Int): Option[SmtpServer] = {
    try {
      repositoryService.listEntities[SmtpServer](typedSearchParameters[SmtpServer]).asScala
        .find(_.get$internalId() == smtpId.toInt)
        .orElse {
          logger.info(s"SMTP server with ID: $smtpId not found. Hence skipping PAT notification.")
          None
        }
    } catch {
      case _: com.xebialabs.deployit.exception.NotFoundException =>
        logger.info(s"SMTP server with ID: $smtpId not found. Hence skipping PAT notification.")
        // Setting isNotified as false when SMTP server is not found
        userTokenService.updateTokenExpiredNotified(tokenId, java.sql.Timestamp.from(Instant.now()), isNotified = false)
        None
    }
  }

  private def getGeneralSettings(): Map[String, String] = {
    configurationRepository.getConfigurationByKey
      .filter(conf => conf.key.endsWith("logo_file") || conf.key.endsWith("pat_email_notification_smtp_server_ci_ref"))
      .map(conf => conf.key.split("\\.").last -> conf.value)
      .toMap
  }

  private def formatExpiryDate(expiryDate: Date): String = {
    val today = LocalDate.now(ZoneId.of("UTC"))
    val expiry = expiryDate.toInstant.atZone(ZoneId.of("UTC")).toLocalDate
    val daysBetween = ChronoUnit.DAYS.between(today, expiry)
    if (daysBetween <= 0) {
      "today"
    } else {
      s"in $daysBetween day(s)"
    }
  }
}
