package ai.digital.deploy.permissions.configuration

import org.hibernate.envers.configuration.EnversSettings
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.boot.autoconfigure.orm.jpa.HibernateSettings
import org.springframework.context.annotation.{Bean, Configuration}
import org.springframework.core.env.Environment
import org.springframework.dao.annotation.PersistenceExceptionTranslationPostProcessor
import org.springframework.data.jpa.repository.config.EnableJpaRepositories
import org.springframework.orm.jpa.vendor.HibernateJpaVendorAdapter
import org.springframework.orm.jpa.{AbstractEntityManagerFactoryBean, JpaTransactionManager, LocalContainerEntityManagerFactoryBean}
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.EnableTransactionManagement

import java.util.Properties
import javax.sql.DataSource

@EnableTransactionManagement
@Configuration
@Autowired
@EnableJpaRepositories(
  basePackages = Array("ai.digital.deploy.permissions.repository.jpa", "ai.digital.deploy.permissions.repository.model"),
  transactionManagerRef = "permissionTransactionManager",
  entityManagerFactoryRef = "permissionEntityManagerFactory",
  enableDefaultTransactions = false
)
class PermissionServiceJpaConfig(env: Environment, @Qualifier("permissionDataSource") permissionDataSource: DataSource) {
  @Bean
  def permissionEntityManagerFactory(): AbstractEntityManagerFactoryBean = {
    val em = new LocalContainerEntityManagerFactoryBean
    em.setDataSource(permissionDataSource)
    em.setPackagesToScan("ai.digital.deploy.permissions.model")
    val vendorAdapter = new HibernateJpaVendorAdapter
    em.setJpaVendorAdapter(vendorAdapter)
    em.setJpaProperties(hibernateProperties)
    em
  }

  private def getSpringJpaProperty(confPath: String, default: String): String =
    env.getProperty("spring.jpa.properties." + confPath, default)

  def hibernateProperties(): Properties = {
    val properties = new Properties()
    properties.put(
      EnversSettings.AUDIT_STRATEGY,
      getSpringJpaProperty(EnversSettings.AUDIT_STRATEGY, "org.hibernate.envers.strategy.internal.ValidityAuditStrategy")
    )
    properties.put(EnversSettings.AUDIT_TABLE_SUFFIX, getSpringJpaProperty(EnversSettings.AUDIT_TABLE_SUFFIX, "_audit"))
    properties.put(EnversSettings.STORE_DATA_AT_DELETE, getSpringJpaProperty(EnversSettings.STORE_DATA_AT_DELETE, "true"))
    properties.put("hibernate.id.new_generator_mappings", getSpringJpaProperty("hibernate.id.new_generator_mappings", "false"))
    properties.put("hibernate.dialect", JpaDialect.initializeDialect(permissionDataSource).dialect)
    properties
  }

  @Bean
  def permissionTransactionManager: PlatformTransactionManager = {
    val transactionManager = new JpaTransactionManager()
    transactionManager.setEntityManagerFactory(permissionEntityManagerFactory().getObject)
    transactionManager.setDataSource(permissionDataSource)
    transactionManager
  }

  @Bean
  def permissionExceptionTranslation = new PersistenceExceptionTranslationPostProcessor
}
