package ai.digital.deploy.permissions.repository.jpa

import ai.digital.deploy.permissions.jpa.TransactionalPermissionRepository
import ai.digital.deploy.permissions.model.{ReferencedPermission, Role}
import ai.digital.deploy.permissions.repository.ReferencedPermissionRepository
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.data.jpa.repository.JpaRepository
import org.springframework.stereotype.Repository

import java.util.{UUID, List => JList}
import scala.jdk.CollectionConverters._

@Repository
@TransactionalPermissionRepository
trait ReferencedPermissionRepositoryImpl extends ReferencedPermissionRepository with JpaRepository[ReferencedPermission, UUID] {
  override def get(roleName: String): List[ReferencedPermission] = findByRoleName(roleName).asScala.toList
  override def get(roleNames: List[String]): List[ReferencedPermission] = findByRoleNameIn(roleNames.asJava).asScala.toList
  override def get(reference: UUID): List[ReferencedPermission] = findByReference(reference).asScala.toList
  override def get(reference: UUID, role: Role): List[ReferencedPermission] =
    get(reference, role.name)
  override def get(reference: UUID, roleName: String): List[ReferencedPermission] =
    findByReferenceAndRoleName(reference, roleName).asScala.toList
  override def getByPermissionsAndRoles(reference: UUID,
                                        permissions: List[String],
                                        roleNames: Set[String]
  ): List[ReferencedPermission] =
    findByReferenceAndPermissionNameInAndRoleNameIn(reference, permissions.asJava, roleNames.toList.asJava).asScala.toList
  override def add(referencedPermission: ReferencedPermission): ReferencedPermission = save(referencedPermission)
  override def add(referencedPermissions: List[ReferencedPermission]): List[ReferencedPermission] =
    saveAll(referencedPermissions.asJava).asScala.toList
  override def list(pageable: Pageable): Page[ReferencedPermission] = findAll(pageable)
  override def listByRoleNamePattern(roleNamePattern: String, reference: UUID, pageable: Pageable): Page[ReferencedPermission] =
    Option(reference)
      .map(r => findByReferenceAndRoleNameContainingIgnoreCase(r, roleNamePattern, pageable))
      .getOrElse(findByRoleNameContainingIgnoreCase(roleNamePattern, pageable))

  override def listByRoleNamePattern(roleNamePattern: String, reference: UUID): List[ReferencedPermission] =
    findByReferenceAndRoleNameContainingIgnoreCase(reference, roleNamePattern).asScala.toList

  override def listByRoleAndPermissions(roleId: UUID, permissionNames: List[String]): List[ReferencedPermission] =
    findByRoleIdAndPermissionNameIn(roleId, permissionNames.asJava).asScala.toList

  override def listByRoleNameAndPermissions(roleName: String, permissionNames: List[String]): List[ReferencedPermission] =
    findByRoleNameAndPermissionNameIn(roleName, permissionNames.asJava).asScala.toList

  override def listByPrincipalNameAndPermissions(principalName: String,
                                                 permissionNames: List[String]
  ): List[ReferencedPermission] =
    findDistinctByRole_Principals_PrincipalNameAndPermissionNameIn(principalName, permissionNames.asJava).asScala.toList

  override def remove(referencedPermission: ReferencedPermission): Unit = delete(referencedPermission)
  override def remove(referencedPermissions: List[ReferencedPermission]): Unit = deleteAll(referencedPermissions.asJava)
  override def removeForRole(role: Role): Unit = deleteAllByRole(role)
  override def removeForReference(referenceId: UUID): Unit = deleteAllByReference(referenceId)
  override def removeAll(): Unit = deleteAll()
  protected def findByReferenceAndRoleName(reference: UUID, name: String): JList[ReferencedPermission]
  protected def findByReference(reference: UUID): JList[ReferencedPermission]
  protected def findByReferenceAndRoleNameContainingIgnoreCase(reference: UUID,
                                                               name: String,
                                                               pageable: Pageable
  ): Page[ReferencedPermission]
  protected def findByReferenceAndRoleNameContainingIgnoreCase(reference: UUID, name: String): JList[ReferencedPermission]
  protected def findByRoleNameContainingIgnoreCase(name: String, pageable: Pageable): Page[ReferencedPermission]

  protected def findByRoleName(roleName: String): JList[ReferencedPermission]
  protected def findByRoleNameIn(roleNames: JList[String]): JList[ReferencedPermission]
  protected def findByRoleIdAndPermissionNameIn(roleId: UUID, permissionNames: JList[String]): JList[ReferencedPermission]
  protected def findByRoleNameAndPermissionNameIn(roleName: String, permissionNames: JList[String]): JList[ReferencedPermission]
  protected def findDistinctByRole_Principals_PrincipalNameAndPermissionNameIn(
    principalName: String,
    permissionNames: JList[String]
  ): JList[ReferencedPermission]
  protected def findByReferenceAndPermissionNameInAndRoleNameIn(reference: UUID,
                                                                names: JList[String],
                                                                roleNames: JList[String]
  ): JList[ReferencedPermission]
  protected def deleteAllByRole(role: Role): Unit
  protected def deleteAllByReference(reference: UUID): Unit
}
