package ai.digital.deploy.permissions.repository.jpa

import ai.digital.deploy.permissions.model.view.{RoleHistory, RoleHistoryId}
import ai.digital.deploy.permissions.repository.RoleHistoryRepository
import ai.digital.deploy.permissions.model.view.RoleHistory
import RoleHistoryRepositoryImpl.{FindAll, FindByRoleQuery}
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.data.jpa.repository.{JpaRepository, Query}
import org.springframework.data.repository.query.Param

import java.time.LocalDateTime
import java.util.{UUID, List => JList}
import scala.jdk.CollectionConverters._

object RoleHistoryRepositoryImpl {
  final val FindByRoleQuery =
    "select h from RoleHistory h where h.roleId = :roleId and (h.startTime between :startTime and :endTime or h.endTime between :startTime and :endTime)"
  final val FindAll =
    "select h from RoleHistory h where (h.startTime between :startTime and :endTime or h.endTime between :startTime and :endTime)"
}

trait RoleHistoryRepositoryImpl extends RoleHistoryRepository with JpaRepository[RoleHistory, RoleHistoryId] {
  override def get(roleId: UUID): List[RoleHistory] = findAllByRoleId(roleId).asScala.toList
  override def getRoleHistoryBetween(roleId: UUID, startTime: LocalDateTime, endTime: LocalDateTime): List[RoleHistory] =
    findAllByRoleIdAndForTimeRange(roleId, startTime, endTime).asScala.toList
  override def getRoleHistories(startTime: LocalDateTime, endTime: LocalDateTime): List[RoleHistory] =
    findAllForTimeRange(startTime, endTime).asScala.toList
  override def findRoleHistories(roleId: UUID,
                                 startTime: LocalDateTime,
                                 endTime: LocalDateTime,
                                 pageable: Pageable
  ): Page[RoleHistory] =
    findAllByRoleIdAndForTimeRange(roleId, startTime, endTime, pageable)
  override def findAllRoleHistories(startTime: LocalDateTime, endTime: LocalDateTime, pageable: Pageable): Page[RoleHistory] =
    findAllForTimeRange(startTime, endTime, pageable)

  protected def findAllByRoleId(roleId: UUID): JList[RoleHistory]

  @Query(FindByRoleQuery)
  protected def findAllByRoleIdAndForTimeRange(@Param("roleId") roleId: UUID,
                                               @Param("startTime") startTime: LocalDateTime,
                                               @Param("endTime") endTime: LocalDateTime
  ): JList[RoleHistory]

  @Query(FindAll)
  protected def findAllForTimeRange(@Param("startTime") startTime: LocalDateTime,
                                    @Param("endTime") endTime: LocalDateTime
  ): JList[RoleHistory]

  @Query(FindByRoleQuery)
  protected def findAllByRoleIdAndForTimeRange(@Param("roleId") roleId: UUID,
                                               @Param("startTime") startTime: LocalDateTime,
                                               @Param("endTime") endTime: LocalDateTime,
                                               pageable: Pageable
  ): Page[RoleHistory]

  @Query(FindAll)
  protected def findAllForTimeRange(@Param("startTime") startTime: LocalDateTime,
                                    @Param("endTime") endTime: LocalDateTime,
                                    pageable: Pageable
  ): Page[RoleHistory]
}
