package ai.digital.deploy.permissions.service.impl

import ai.digital.deploy.permissions.exception.RoleNameNotFoundException
import ai.digital.deploy.permissions.jpa.TransactionalPermissionService
import ai.digital.deploy.permissions.model.{GlobalPermission, Role}
import ai.digital.deploy.permissions.repository.GlobalPermissionRepository
import ai.digital.deploy.permissions.service.{GlobalPermissionService, RolePrincipalService, RoleService}
import org.springframework.stereotype.Service

import java.util.UUID

@Service
@TransactionalPermissionService
class GlobalPermissionServiceImpl(globalPermissionRepository: GlobalPermissionRepository,
                                  roleService: RoleService,
                                  rolePrincipalService: RolePrincipalService
) extends GlobalPermissionService {
  override def add(roleName: String, permissions: List[String]): (Role, List[GlobalPermission]) = {
    val role: Role = getRole(roleName)
    val (exists, toDelete) = globalPermissionRepository.get(role).partition(gp => permissions.contains(gp.permissionName))
    val toSave = permissions
      .filterNot { p =>
        exists.map(_.permissionName).contains(p)
      }.map(p => GlobalPermission(role, p))
    globalPermissionRepository.remove(toDelete)
    (role, exists ++ globalPermissionRepository.add(toSave))
  }

  override def remove(roleName: String, permissions: List[String]): (Role, List[GlobalPermission]) = {
    val role: Role = getRole(roleName)
    val (toDelete, gp) =
      globalPermissionRepository.get(role).partition(gp => permissions.contains(gp.permissionName))
    globalPermissionRepository.remove(toDelete)
    (role, gp)
  }

  override def removeForRole(role: Role): Unit =
    globalPermissionRepository.removeForRole(role)

  override def removeAll(): Unit = globalPermissionRepository.removeAll()

  override def read(): List[GlobalPermission] =
    globalPermissionRepository.get()

  override def read(roleName: String): List[GlobalPermission] =
    globalPermissionRepository.get(roleName)

  override def read(roleNames: List[String]): List[GlobalPermission] =
    globalPermissionRepository.get(roleNames)

  override def readByRolePattern(rolePattern: String): List[GlobalPermission] =
    wrapNullablePatternSearch(
      rolePattern,
      p => globalPermissionRepository.listByRoleNamePattern(p),
      () => globalPermissionRepository.get()
    )

  override def edit(roleName: String,
                    permissionsToAdd: List[String],
                    permissionsToDelete: List[String]
  ): (Role, List[GlobalPermission]) = {
    val role: Role = getRole(roleName)
    val permissions = globalPermissionRepository.get(role)
    val (toDelete, exists) = permissions.partition(p => permissionsToDelete.contains(p.permissionName))
    globalPermissionRepository.remove(toDelete)
    val toAdd = permissionsToAdd.map(p => GlobalPermission(role, p))
    (role, exists ++ globalPermissionRepository.add(toAdd))
  }

  private def getRole(roleName: String) = {
    val role =
      roleService.read(roleName).getOrElse(throw RoleNameNotFoundException(roleName))
    role
  }

  override def getGlobalPermissionsForRoleId(roleId: UUID): List[String] = {
    val permissions = globalPermissionRepository.get(roleId)
    permissions.map(_.permissionName)
  }

  override def getGlobalPermissionsForPrincipal(principal: String): Set[String] = {
    val result = rolePrincipalService.read(principal)
    val roles = result.map(_.role.name)
    globalPermissionRepository.get(roles).map(data => data.permissionName).toSet
  }

  override def checkPermission(permissions: List[String], allRoles: List[String]): Boolean =
    globalPermissionRepository.getByPermissionsAndRoles(permissions, allRoles.toSet).nonEmpty

  override def checkPermission(permissions: List[String], allRoles: List[String], principals: List[String]): Boolean = {
    val principalRoleIds = rolePrincipalService.readAll(principals).map(_.role.name).toSet
    globalPermissionRepository.getByPermissionsAndRoles(permissions, principalRoleIds ++ allRoles).nonEmpty
  }
}
