package ai.digital.deploy.permissions.service.impl

import ai.digital.deploy.permissions.exception.RoleNameNotFoundException
import ai.digital.deploy.permissions.jpa.TransactionalPermissionService
import ai.digital.deploy.permissions.model.view.RolePrincipalReferenceView
import ai.digital.deploy.permissions.model.{Role, RolePrincipal}
import ai.digital.deploy.permissions.repository.{RolePrincipalReferenceViewRepository, RolePrincipalRepository}
import ai.digital.deploy.permissions.service.{RolePrincipalService, RoleService}
import org.springframework.data.domain.{Page, PageImpl, Pageable}
import org.springframework.stereotype.Service

import java.util.UUID
import scala.jdk.CollectionConverters._

@Service
@TransactionalPermissionService
class RolePrincipalServiceImpl(rolePrincipalRepository: RolePrincipalRepository,
                               rolePrincipalReferenceViewRepository: RolePrincipalReferenceViewRepository,
                               roleService: RoleService
) extends RolePrincipalService {
  override def add(roleName: String, principals: List[String]): (Role, List[RolePrincipal]) = {
    val role: Role = getRole(roleName)
    val (exists, toDelete) = rolePrincipalRepository.get(role).partition(rp => principals.contains(rp.principalName))
    val toSave = principals
      .filterNot { p =>
        exists.map(_.principalName).contains(p)
      }.map(p => RolePrincipal(role, p))
    rolePrincipalRepository.remove(toDelete)
    (role, exists ++ rolePrincipalRepository.add(toSave))
  }

  override def remove(roleName: String, principals: List[String]): (Role, List[RolePrincipal]) = {
    val role: Role = getRole(roleName)
    val (toDelete, rp) = rolePrincipalRepository.get(role).partition(rp => principals.contains(rp.principalName))
    rolePrincipalRepository.remove(toDelete)
    (role, rp)
  }

  override def removeForRole(role: Role): Unit =
    rolePrincipalRepository.removeForRole(role)

  override def read(roleName: String): List[RolePrincipal] = {
    val role: Role = getRole(roleName)
    rolePrincipalRepository.get(role)
  }

  override def read(roleNamePattern: String, pageable: Pageable): (Page[Role], List[RolePrincipal]) = {
    val rolePage = roleService.read(roleNamePattern, pageable)
    val roleIds = rolePage.getContent.asScala.toList.map(_.id)
    val views = rolePrincipalRepository.get().filter(rp => roleIds.contains(rp.role.id))
    (rolePage, views)
  }

  override def read(principalName: String, roleNamePattern: String, pageable: Pageable): Page[RolePrincipal] =
    wrapNullablePatternSearch(
      roleNamePattern,
      p => rolePrincipalRepository.listByPrincipalNameAndRoleNamePattern(principalName, p, pageable),
      () => rolePrincipalRepository.listByPrincipalName(principalName, pageable)
    )

  override def read(referenceId: UUID,
                    roleNamePattern: String,
                    pageable: Pageable
  ): (Page[Role], List[RolePrincipalReferenceView]) = {
    val rolePage = roleService.read(roleNamePattern, pageable)
    val roleIds = rolePage.getContent.asScala.toList.map(_.id)
    val views = rolePrincipalReferenceViewRepository.get(roleIds, referenceId)
    val roleIdsOnReference = views.map(_.roleId).distinct
    val rolesOnReference = rolePage.getContent.asScala.toList.filter(role => roleIdsOnReference.contains(role.id))
    val rolePageForReference = new PageImpl[Role](rolesOnReference.asJava, pageable, rolesOnReference.size)
    (rolePageForReference, views)
  }

  override def readAll(principals: List[String]): List[RolePrincipal] =
    rolePrincipalRepository.getAll(principals)

  override def edit(roleName: String,
                    principalsToAdd: List[String],
                    principalsToDelete: List[String]
  ): (Role, List[RolePrincipal]) = {
    val role: Role = getRole(roleName)
    val (toDelete, exists) = rolePrincipalRepository.get(role).partition(p => principalsToDelete.contains(p.principalName))
    rolePrincipalRepository.remove(toDelete)
    val toAdd = principalsToAdd.map(p => RolePrincipal(role, p))
    (role, exists ++ rolePrincipalRepository.add(toAdd))
  }

  private def getRole(roleName: String) = {
    val role =
      roleService.read(roleName).getOrElse(throw RoleNameNotFoundException(roleName))
    role
  }

  override def read(principals: List[String], roleNamePattern: String, pageable: Pageable): Page[RolePrincipal] =
    rolePrincipalRepository.listByPrincipalNamesAndRoleNamePattern(principals, roleNamePattern, pageable)

  override def readByRolePattern(rolePattern: String): List[RolePrincipal] =
    wrapNullablePatternSearch(
      rolePattern,
      p => rolePrincipalRepository.getByRolePattern(p),
      () => rolePrincipalRepository.get()
    )

  override def read(principal: String, rolePattern: String): List[RolePrincipal] =
    wrapNullablePatternSearch(
      rolePattern,
      p => rolePrincipalRepository.getByPrincipalAndRolePattern(principal, p),
      () => rolePrincipalRepository.get(principal)
    )

  override def read(principals: List[String], rolePattern: String): List[RolePrincipal] =
    wrapNullablePatternSearch(
      rolePattern,
      p => rolePrincipalRepository.getByPrincipalsAndRolePattern(principals, p),
      () => rolePrincipalRepository.getAll(principals)
    )
}
