package ai.digital.deploy.permissions.service.impl

import ai.digital.deploy.permissions.exception.{RoleNameAlreadyExistsException, RoleNameNotFoundException}
import ai.digital.deploy.permissions.jpa.TransactionalPermissionService
import ai.digital.deploy.permissions.model.Role
import ai.digital.deploy.permissions.repository.{GlobalPermissionRepository, ReferencedPermissionRepository, RolePrincipalRepository, RoleRepository}
import ai.digital.deploy.permissions.service.RoleService
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.stereotype.Service

import java.util.UUID

@Service
@TransactionalPermissionService
class RoleServiceImpl(roleRepository: RoleRepository,
                      rolePrincipalRepository: RolePrincipalRepository,
                      globalPermissionRepository: GlobalPermissionRepository,
                      referencedPermissionRepository: ReferencedPermissionRepository
) extends RoleService {
  override def create(name: String): Role = roleRepository.add(Role(name = name))
  override def createOrUpdate(role: Role): Role = {
    if (roleRepository.get(role.id).isEmpty)
      if (roleRepository.get(role.name).isDefined)
        throw RoleNameAlreadyExistsException(role.name)
    roleRepository.add(role)
  }

  override def delete(name: String): Unit = roleRepository.removeByName(name)
  override def delete(id: UUID): Unit = roleRepository.removeById(id)
  override def deleteAllRoleReferences(roleName: String): Unit = {
    val role = read(roleName)
      .getOrElse(throw RoleNameNotFoundException(roleName))
    rolePrincipalRepository.removeForRole(role)
    globalPermissionRepository.removeForRole(role)
    referencedPermissionRepository.removeForRole(role)
    remove(role)
  }
  override def remove(role: Role): Unit = roleRepository.remove(role)
  override def removeAll(): Unit = {
    rolePrincipalRepository.removeAll()
    roleRepository.removeAll()
  }
  override def update(role: Role): Role = roleRepository.add(role)
  override def read(name: String): Option[Role] =
    roleRepository.get(name)
  override def read(names: List[String]): List[Role] = {
    val roles = roleRepository.get(names)
    if (roles.size != names.size)
      throw RoleNameNotFoundException((names diff roles.map(_.name)).mkString("[", ", ", "]"))
    roles
  }
  override def read(names: List[String], pageable: Pageable): Page[Role] =
    roleRepository.get(names, pageable)

  override def read(id: UUID): Option[Role] =
    roleRepository.get(id)

  override def read(namePattern: String, pageable: Pageable): Page[Role] =
    wrapNullablePatternSearch(
      namePattern,
      p => roleRepository.listByNamePattern(p, pageable),
      () => roleRepository.list(pageable)
    )

  override def readByRolePattern(namePattern: String): List[Role] =
    wrapNullablePatternSearch(
      namePattern,
      p => roleRepository.listByNamePattern(p),
      () => roleRepository.getAll()
    )

  override def readAll(): List[Role] = roleRepository.getAll()
  override def rename(name: String, newName: String): Role = {
    val role = read(name).getOrElse(throw RoleNameNotFoundException(name))
    roleRepository.add(Role(role.id, newName))
  }

  override def count(roleNamePattern: String): Long =
    wrapNullablePatternSearch(
      roleNamePattern,
      p => roleRepository.countRolesByRoleNamePattern(p),
      () => roleRepository.countAllRoles()
    )
}
