package ai.digital.deploy.permissions.model

import org.hibernate.annotations.{Cache, CacheConcurrencyStrategy}
import org.hibernate.envers.{Audited, NotAudited}

import java.util.{UUID, List => JList}
import javax.persistence.{Column, Entity, OneToMany, Table}
import scala.jdk.CollectionConverters._

@Entity
@Table(name = "ps_role")
@Audited
class Role extends BaseModel {
  @Column(name = "name", unique = true, nullable = false)
  var name: String = _

  @OneToMany(mappedBy = "role")
  @Cache(usage = CacheConcurrencyStrategy.NONSTRICT_READ_WRITE)
  @NotAudited
  var principals: JList[RolePrincipal] = _

  @OneToMany(mappedBy = "role")
  @NotAudited
  var referencedPermissions: JList[ReferencedPermission] = _

  @OneToMany(mappedBy = "role")
  @NotAudited
  var globalPermissions: JList[GlobalPermission] = _

  override def toString = s"Role($id, $name)"

  def canEqual(other: Any): Boolean =
    other match {
      case _: Role => true
      case _ => false
    }

  override def equals(other: Any): Boolean =
    other match {
      case that: Role =>
        (that canEqual this) &&
          name == that.name
      case _ => false
    }

  override def hashCode(): Int = {
    val state = Seq(name)
    state.map(_.hashCode()).foldLeft(0)((a, b) => 31 * a + b)
  }
}

object Role {
  def apply(name: String): Role = {
    val role = new Role
    role.name = name
    role
  }

  def apply(id: UUID, name: String): Role = {
    val role = new Role
    role.name = name
    role.id = id
    role
  }

  def apply(id: UUID,
            name: String,
            referencedPermissions: List[ReferencedPermission],
            globalPermissions: List[GlobalPermission]
  ): Role = {
    val role = new Role
    role.id = id
    role.name = name
    role.referencedPermissions = referencedPermissions.asJava
    role.globalPermissions = globalPermissions.asJava
    role
  }

  def apply(id: UUID,
            name: String,
            principals: List[RolePrincipal],
            referencedPermissions: List[ReferencedPermission],
            globalPermissions: List[GlobalPermission]
  ): Role = {
    val role = new Role
    role.id = id
    role.name = name
    role.principals = principals.asJava
    role.referencedPermissions = referencedPermissions.asJava
    role.globalPermissions = globalPermissions.asJava
    role
  }
}
