package ai.digital.deploy.permissions.service.impl

import ai.digital.deploy.permissions.jpa.{ReadOnlyTransactionalPermissionService, TransactionalPermissionService}
import ai.digital.deploy.permissions.model.Role
import ai.digital.deploy.permissions.repository.{GlobalPermissionRepository, ReferencedPermissionRepository, RolePrincipalRepository, RoleRepository}
import ai.digital.deploy.permissions.service.{RoleNameAlreadyExistsServiceException, RoleNameNotFoundServiceException, RoleService}
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.stereotype.Service

import java.util.UUID
import scala.jdk.CollectionConverters._

@Service
@TransactionalPermissionService
class RoleServiceImpl(roleRepository: RoleRepository,
                      rolePrincipalRepository: RolePrincipalRepository,
                      globalPermissionRepository: GlobalPermissionRepository,
                      referencedPermissionRepository: ReferencedPermissionRepository
) extends RoleService {
  override def create(name: String): Either[RoleNameAlreadyExistsServiceException, Role] =
    if (roleRepository.get(name).isDefined)
      Left(RoleNameAlreadyExistsServiceException(name))
    else
      Right(roleRepository.add(Role(name)))

  override def createOrUpdate(role: Role): Either[RoleNameAlreadyExistsServiceException, Role] =
    if (roleRepository.get(role.id).isEmpty)
      if (roleRepository.get(role.name).isDefined)
        Left(RoleNameAlreadyExistsServiceException(role.name))
      else
        Right(roleRepository.add(role))
    else
      Right(roleRepository.update(role))

  override def delete(name: String): Either[RoleNameNotFoundServiceException, Unit] =
    read(name)
      .map(role => Right(roleRepository.remove(role))).getOrElse(
        Left(RoleNameNotFoundServiceException(name))
      )

  override def delete(id: UUID): Unit = roleRepository.removeById(id)
  override def deleteAllRoleReferences(roleName: String): Either[RoleNameNotFoundServiceException, Unit] =
    read(roleName)
      .map { role =>
        rolePrincipalRepository.removeForRole(role)
        globalPermissionRepository.removeForRole(role)
        referencedPermissionRepository.removeForRole(role)
        Right(remove(role))
      }.getOrElse(Left(RoleNameNotFoundServiceException(roleName)))

  override def remove(role: Role): Unit = roleRepository.remove(role)
  override def removeAll(): Unit = {
    rolePrincipalRepository.removeAll()
    roleRepository.removeAll()
  }
  override def update(role: Role): Role = roleRepository.add(role)

  @ReadOnlyTransactionalPermissionService
  override def read(name: String): Option[Role] =
    roleRepository.get(name)

  @ReadOnlyTransactionalPermissionService
  override def read(names: List[String]): Either[RoleNameNotFoundServiceException, List[Role]] = {
    val roles = roleRepository.get(names)
    if (roles.size != names.size)
      Left(RoleNameNotFoundServiceException((names diff roles.map(_.name)).mkString("[", ", ", "]")))
    else
      Right(roles)
  }

  @ReadOnlyTransactionalPermissionService
  override def read(names: List[String], pageable: Pageable): Page[Role] =
    roleRepository.get(names, pageable)

  @ReadOnlyTransactionalPermissionService
  override def read(id: UUID): Option[Role] =
    roleRepository.get(id).map { r =>
      r.principals = rolePrincipalRepository.get(r).asJava
      r
    }

  @ReadOnlyTransactionalPermissionService
  override def read(namePattern: String, pageable: Pageable): Page[Role] =
    wrapNullablePatternSearch(
      namePattern,
      p => roleRepository.listByNamePattern(p, pageable),
      () => roleRepository.list(pageable)
    )

  @ReadOnlyTransactionalPermissionService
  override def readByRolePattern(namePattern: String): List[Role] =
    wrapNullablePatternSearch(
      namePattern,
      p => roleRepository.listByNamePattern(p),
      () => roleRepository.getAll()
    )

  @ReadOnlyTransactionalPermissionService
  override def readAll(): List[Role] = roleRepository.getAll()
  override def rename(name: String, newName: String): Either[Throwable, Role] =
    read(name)
      .map(role =>
        if (read(newName).isDefined)
          Left(RoleNameAlreadyExistsServiceException(newName))
        else
          Right(roleRepository.update(Role(role.id, newName)))
      ).getOrElse(Left(RoleNameNotFoundServiceException(name)))

  @ReadOnlyTransactionalPermissionService
  override def count(roleNamePattern: String): Long =
    wrapNullablePatternSearch(
      roleNamePattern,
      p => roleRepository.countRolesByRoleNamePattern(p),
      () => roleRepository.countAllRoles()
    )
}
