package ai.digital.deploy.permissions.service.impl

import ai.digital.deploy.permissions.jpa.{ReadOnlyTransactionalPermissionService, TransactionalPermissionService}
import ai.digital.deploy.permissions.model.view.RoleReferencedPermissionView
import ai.digital.deploy.permissions.model.{ReferencedPermission, Role}
import ai.digital.deploy.permissions.repository.{ReferencedPermissionRepository, RoleReferencedPermissionViewRepository}
import ai.digital.deploy.permissions.service.{ReferencedPermissionService, RoleNameNotFoundServiceException, RolePrincipalService, RoleService}
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.stereotype.Service

import java.util.UUID
import scala.jdk.CollectionConverters._

@Service
@TransactionalPermissionService
class ReferencedPermissionServiceImpl(referencedPermissionRepository: ReferencedPermissionRepository,
                                      roleService: RoleService,
                                      rolePrincipalService: RolePrincipalService,
                                      roleReferencedPermissionViewRepository: RoleReferencedPermissionViewRepository
) extends ReferencedPermissionService {
  override def add(referenceId: UUID,
                   roleName: String,
                   permissions: List[String]
  ): Either[RoleNameNotFoundServiceException, (Role, List[ReferencedPermission])] =
    roleService
      .read(roleName).map { role =>
        val savedPermissions = referencedPermissionRepository.get(referenceId, role)
        val (exists, toDelete) = savedPermissions.partition(rp => permissions.contains(rp.permissionName))
        val toSave = permissions
          .filterNot { p =>
            exists.map(_.permissionName).contains(p)
          }.map(p => ReferencedPermission(role, p, referenceId))
        referencedPermissionRepository.remove(toDelete)
        Right((role, exists ++ referencedPermissionRepository.add(toSave)))
      }.getOrElse(Left(RoleNameNotFoundServiceException(roleName)))

  override def remove(referenceId: UUID,
                      roleName: String,
                      permissions: List[String]
  ): Either[RoleNameNotFoundServiceException, (Role, List[ReferencedPermission])] =
    roleService
      .read(roleName).map { role =>
        val (toDelete, rp) =
          referencedPermissionRepository.get(referenceId, role).partition(rp => permissions.contains(rp.permissionName))
        referencedPermissionRepository.remove(toDelete)
        Right((role, rp))
      }.getOrElse(Left(RoleNameNotFoundServiceException(roleName)))

  override def removeForRole(role: Role): Unit = referencedPermissionRepository.removeForRole(role)

  override def removeForReference(referenceId: UUID): Unit = referencedPermissionRepository.removeForReference(referenceId)

  override def removeAll(): Unit = referencedPermissionRepository.removeAll()

  @ReadOnlyTransactionalPermissionService
  override def read(roleName: String): Either[RoleNameNotFoundServiceException, List[ReferencedPermission]] =
    roleService
      .read(roleName).map(role => Right(referencedPermissionRepository.get(role.name))).getOrElse(
        Left(RoleNameNotFoundServiceException(roleName))
      )

  @ReadOnlyTransactionalPermissionService
  override def read(roleNames: List[String]): Either[RoleNameNotFoundServiceException, List[ReferencedPermission]] =
    roleService.read(roleNames) match {
      case Left(exception) => Left(exception)
      case Right(_) => Right(referencedPermissionRepository.get(roleNames))
    }

  @ReadOnlyTransactionalPermissionService
  override def read(referenceId: UUID, roleName: String): Either[RoleNameNotFoundServiceException, List[ReferencedPermission]] =
    roleService
      .read(roleName).map(role => Right(referencedPermissionRepository.get(referenceId, role.name))).getOrElse(
        Left(RoleNameNotFoundServiceException(roleName))
      )

  @ReadOnlyTransactionalPermissionService
  override def read(referenceId: UUID,
                    rolePattern: String,
                    pageable: Pageable
  ): (Page[Role], List[RoleReferencedPermissionView]) = {
    val rolePage = roleService.read(rolePattern, pageable)
    val roleIds = rolePage.getContent.asScala.toList.map(_.id)
    val views = roleReferencedPermissionViewRepository.get(roleIds, referenceId)
    (rolePage, views)
  }

  @ReadOnlyTransactionalPermissionService
  override def readForRolePattern(referenceId: UUID, rolePattern: String): List[ReferencedPermission] =
    wrapNullablePatternSearch(
      rolePattern,
      p => referencedPermissionRepository.listByRoleNamePattern(p, referenceId),
      () => referencedPermissionRepository.get(referenceId)
    )

  override def edit(referenceId: UUID,
                    roleName: String,
                    permissionsToAdd: List[String],
                    permissionToDelete: List[String]
  ): Either[RoleNameNotFoundServiceException, (Role, List[ReferencedPermission])] =
    roleService
      .read(roleName).map { role =>
        val permissions = referencedPermissionRepository.get(referenceId, role)
        val (toDelete, exists) = permissions.partition(p => permissionToDelete.contains(p.permissionName))
        referencedPermissionRepository.remove(toDelete)
        val toAdd = permissionsToAdd.map(p => ReferencedPermission(role, p, referenceId))
        Right((role, exists ++ referencedPermissionRepository.add(toAdd)))
      }.getOrElse(Left(RoleNameNotFoundServiceException(roleName)))

  @ReadOnlyTransactionalPermissionService
  override def getReferencesForRole(roleId: UUID, permissions: List[String]): Set[UUID] =
    referencedPermissionRepository.listByRoleAndPermissions(roleId, permissions).map(_.reference).toSet

  @ReadOnlyTransactionalPermissionService
  override def getReferencesForRoles(roleIds: List[UUID], permissions: List[String]): Set[UUID] =
    referencedPermissionRepository.listByRolesAndPermissions(roleIds, permissions).map(_.reference).toSet

  @ReadOnlyTransactionalPermissionService
  override def getReferencesForRoleName(roleName: String, permissions: List[String]): Set[UUID] =
    referencedPermissionRepository.listByRoleNameAndPermissions(roleName, permissions).map(_.reference).toSet

  @ReadOnlyTransactionalPermissionService
  override def getReferencesForPrincipalName(principalName: String, permissions: List[String]): Set[UUID] =
    referencedPermissionRepository.listByPrincipalNameAndPermissions(principalName, permissions).map(_.reference).toSet

  @ReadOnlyTransactionalPermissionService
  override def checkPermission(referenceId: UUID, permissions: List[String], allRoles: List[String]): Boolean =
    referencedPermissionRepository.getByPermissionsAndRoles(referenceId, permissions, allRoles.toSet).nonEmpty

  @ReadOnlyTransactionalPermissionService
  override def checkPermission(referenceId: UUID,
                               permissions: List[String],
                               allRoles: List[String],
                               principals: List[String]
  ): Boolean = {
    val principalRoleIds = rolePrincipalService.readAll(principals).map(_.role.name).toSet
    referencedPermissionRepository.getByPermissionsAndRoles(referenceId, permissions, principalRoleIds ++ allRoles).nonEmpty
  }
}
