package ai.digital.deploy.permissions.repository.jpa

import ai.digital.deploy.permissions.jpa.TransactionalPermissionRepository
import ai.digital.deploy.permissions.model.{GlobalPermission, Role}
import ai.digital.deploy.permissions.repository.GlobalPermissionRepository
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.data.jpa.repository.JpaRepository
import org.springframework.stereotype.Repository

import java.util.{List => JList, UUID}
import scala.jdk.CollectionConverters._

@Repository
@TransactionalPermissionRepository
trait GlobalPermissionRepositoryImpl extends GlobalPermissionRepository with JpaRepository[GlobalPermission, UUID] {
  override def get(): List[GlobalPermission] = findAll().asScala.toList
  override def get(role: Role): List[GlobalPermission] = findByRoleName(role.name).asScala.toList
  override def get(roleId: UUID): List[GlobalPermission] = findByRoleId(roleId).asScala.toList
  override def get(roleName: String): List[GlobalPermission] = findByRoleName(roleName).asScala.toList
  override def get(roleNames: List[String]): List[GlobalPermission] = findByRoleNameIn(roleNames.asJava).asScala.toList
  override def getByPermissionsAndRoles(permissions: List[String], roleNames: Set[String]): List[GlobalPermission] =
    findByPermissionNameInAndRoleNameIn(permissions.asJava, roleNames.toList.asJava).asScala.toList
  override def add(globalPermission: GlobalPermission): GlobalPermission = save(globalPermission)
  override def add(globalPermissions: List[GlobalPermission]): List[GlobalPermission] =
    saveAll(globalPermissions.asJava).asScala.toList
  override def list(pageable: Pageable): Page[GlobalPermission] = findAll(pageable)
  override def listByRoleNamePattern(roleNamePattern: String, pageable: Pageable): Page[GlobalPermission] =
    findByRoleNameContainingIgnoreCase(roleNamePattern, pageable)
  override def listByRoleNamePattern(roleNamePattern: String): List[GlobalPermission] =
    findByRoleNameContainingIgnoreCase(roleNamePattern).asScala.toList
  override def remove(globalPermission: GlobalPermission): Unit = delete(globalPermission)
  override def remove(globalPermissions: List[GlobalPermission]): Unit = deleteAll(globalPermissions.asJava)
  override def removeForRole(role: Role): Unit = deleteAllByRole(role)
  override def removeAll(): Unit = deleteAll()
  protected def findByRoleName(name: String): JList[GlobalPermission]
  protected def findByRoleNameIn(names: JList[String]): JList[GlobalPermission]
  protected def findByRoleId(roleId: UUID): JList[GlobalPermission]
  protected def findByRoleNameContainingIgnoreCase(name: String, pageable: Pageable): Page[GlobalPermission]
  protected def findByRoleNameContainingIgnoreCase(name: String): JList[GlobalPermission]
  protected def findByPermissionNameInAndRoleNameIn(names: JList[String], roleNames: JList[String]): JList[GlobalPermission]
  protected def deleteAllByRole(role: Role): Unit
}
