package ai.digital.deploy.permissions.repository.jpa

import ai.digital.deploy.permissions.model.view.{ReferencedPermissionHistory, ReferencedPermissionHistoryId}
import ai.digital.deploy.permissions.repository.ReferencedPermissionHistoryRepository
import ReferencedPermissionHistoryRepositoryImpl._
import org.springframework.data.domain.{Page, Pageable, Sort}
import org.springframework.data.jpa.repository.{JpaRepository, Query}
import org.springframework.data.repository.query.Param

import java.time.LocalDateTime
import java.util.{List => JList, UUID}
import scala.jdk.CollectionConverters._

object ReferencedPermissionHistoryRepositoryImpl {
  final val FindByReferencePermissionIdAndReferenceQuery =
    """select h from ReferencedPermissionHistory h where h.reference = :reference and h.referencedPermissionId = :referencedPermissionId
       and (h.startTime between :startTime and :endTime or h.endTime between :startTime and :endTime)"""
  final val FindByPermissionNameAndReferenceQuery =
    """select h from ReferencedPermissionHistory h where h.reference = :reference and h.permissionName = :permissionName
       and (h.startTime between :startTime and :endTime or h.endTime between :startTime and :endTime)"""
  final val FindByRoleIdAndReferenceQuery =
    """select h from ReferencedPermissionHistory h where h.reference = :reference and h.roleId = :roleId and (h.startTime between :startTime and :endTime or h.endTime between :startTime and :endTime)"""

}

trait ReferencedPermissionHistoryRepositoryImpl
    extends ReferencedPermissionHistoryRepository
    with JpaRepository[ReferencedPermissionHistory, ReferencedPermissionHistoryId] {
  override def get(referencedPermissionId: UUID, reference: UUID): List[ReferencedPermissionHistory] =
    findByReferencedPermissionIdAndReference(referencedPermissionId, reference).asScala.toList
  override def get(permissionName: String, reference: UUID): List[ReferencedPermissionHistory] =
    findByPermissionNameAndReference(permissionName, reference).asScala.toList
  override def getReferencedPermissionHistoriesForRole(roleId: UUID): List[ReferencedPermissionHistory] =
    findByRoleId(roleId).asScala.toList
  override def getReferencedPermissionHistoriesForReferenceRoleBetween(roleId: UUID,
                                                                       reference: UUID,
                                                                       startTime: LocalDateTime,
                                                                       endTime: LocalDateTime
  ): List[ReferencedPermissionHistory] =
    findAllByRoleIdAndReferenceForTimeRange(roleId, reference, startTime, endTime).asScala.toList

  override def getReferencedPermissionHistoriesForRoleAndPermissions(
    roleId: UUID,
    permissionNames: List[String]
  ): List[ReferencedPermissionHistory] = findByRoleIdAndPermissionNameIn(roleId, permissionNames.asJava).asScala.toList

  override def getReferencedPermissionHistoriesForRoleAndReference(
    roleId: UUID,
    reference: UUID,
    permissionNames: List[String]
  ): List[ReferencedPermissionHistory] =
    findByRoleIdAndReferenceAndPermissionNameIn(roleId, reference, permissionNames.asJava).asScala.toList

  override def getReferencedPermissionHistoryBetween(referencedPermissionId: UUID,
                                                     reference: UUID,
                                                     startTime: LocalDateTime,
                                                     endTime: LocalDateTime
  ): List[ReferencedPermissionHistory] =
    findAllByReferencedPermissionIdAndReferenceForTimeRange(referencedPermissionId, reference, startTime, endTime).asScala.toList
  override def getReferencedPermissionHistoryBetween(permissionName: String,
                                                     reference: UUID,
                                                     startTime: LocalDateTime,
                                                     endTime: LocalDateTime
  ): List[ReferencedPermissionHistory] =
    findAllByPermissionNameAndReferenceForTimeRange(permissionName, reference, startTime, endTime).asScala.toList

  override def findAllReferencedPermissionHistoriesForRole(roleId: UUID, pageable: Pageable): Page[ReferencedPermissionHistory] =
    findByRoleId(roleId, pageable)

  override def findAllReferencedPermissionHistoriesForReferenceRoleBetween(
    roleId: UUID,
    reference: UUID,
    startTime: LocalDateTime,
    endTime: LocalDateTime,
    pageable: Pageable
  ): Page[ReferencedPermissionHistory] = findAllByRoleIdAndReferenceForTimeRange(roleId, reference, startTime, endTime, pageable)

  override def findReferencedPermissionHistories(referencedPermissionId: UUID,
                                                 reference: UUID,
                                                 startTime: LocalDateTime,
                                                 endTime: LocalDateTime,
                                                 pageable: Pageable
  ): Page[ReferencedPermissionHistory] =
    findAllByReferencedPermissionIdAndReferenceForTimeRange(referencedPermissionId, reference, startTime, endTime, pageable)
  override def findReferencedPermissionHistories(permissionName: String,
                                                 reference: UUID,
                                                 startTime: LocalDateTime,
                                                 endTime: LocalDateTime,
                                                 pageable: Pageable
  ): Page[ReferencedPermissionHistory] =
    findAllByPermissionNameAndReferenceForTimeRange(permissionName, reference, startTime, endTime, pageable)

  protected def findByReferencedPermissionIdAndReference(referencedPermissionId: UUID,
                                                         reference: UUID
  ): JList[ReferencedPermissionHistory]

  protected def findByPermissionNameAndReference(permissionName: String, reference: UUID): JList[ReferencedPermissionHistory]

  protected def findByRoleIdAndPermissionNameIn(
    roleId: UUID,
    permissionNames: JList[String],
    sort: Sort = Sort.by("startTime")
  ): JList[ReferencedPermissionHistory]

  protected def findByRoleIdAndReferenceAndPermissionNameIn(
    roleId: UUID,
    reference: UUID,
    permissionNames: JList[String],
    sort: Sort = Sort.by("startTime")
  ): JList[ReferencedPermissionHistory]

  protected def findByRoleId(roleId: UUID): JList[ReferencedPermissionHistory]

  protected def findByRoleId(roleId: UUID, pageable: Pageable): Page[ReferencedPermissionHistory]

  @Query(FindByReferencePermissionIdAndReferenceQuery)
  protected def findAllByReferencedPermissionIdAndReferenceForTimeRange(
    @Param("referencedPermissionId") referencedPermissionId: UUID,
    @Param("reference") reference: UUID,
    @Param("startTime") startTime: LocalDateTime,
    @Param("endTime") endTime: LocalDateTime
  ): JList[ReferencedPermissionHistory]

  @Query(FindByReferencePermissionIdAndReferenceQuery)
  protected def findAllByReferencedPermissionIdAndReferenceForTimeRange(
    @Param("referencedPermissionId") referencedPermissionId: UUID,
    @Param("reference") reference: UUID,
    @Param("startTime") startTime: LocalDateTime,
    @Param("endTime") endTime: LocalDateTime,
    pageable: Pageable
  ): Page[ReferencedPermissionHistory]

  @Query(FindByPermissionNameAndReferenceQuery)
  protected def findAllByPermissionNameAndReferenceForTimeRange(
    @Param("permissionName") permissionName: String,
    @Param("reference") reference: UUID,
    @Param("startTime") startTime: LocalDateTime,
    @Param("endTime") endTime: LocalDateTime
  ): JList[ReferencedPermissionHistory]

  @Query(FindByPermissionNameAndReferenceQuery)
  protected def findAllByPermissionNameAndReferenceForTimeRange(@Param("permissionName") permissionName: String,
                                                                @Param("reference") reference: UUID,
                                                                @Param("startTime") startTime: LocalDateTime,
                                                                @Param("endTime") endTime: LocalDateTime,
                                                                pageable: Pageable
  ): Page[ReferencedPermissionHistory]

  @Query(FindByRoleIdAndReferenceQuery)
  protected def findAllByRoleIdAndReferenceForTimeRange(
    @Param("roleId") roleId: UUID,
    @Param("reference") reference: UUID,
    @Param("startTime") startTime: LocalDateTime,
    @Param("endTime") endTime: LocalDateTime
  ): JList[ReferencedPermissionHistory]

  @Query(FindByRoleIdAndReferenceQuery)
  protected def findAllByRoleIdAndReferenceForTimeRange(@Param("roleId") roleId: UUID,
                                                        @Param("reference") reference: UUID,
                                                        @Param("startTime") startTime: LocalDateTime,
                                                        @Param("endTime") endTime: LocalDateTime,
                                                        pageable: Pageable
  ): Page[ReferencedPermissionHistory]
}
