package ai.digital.deploy.permissions.repository.jpa

import ai.digital.deploy.permissions.model.view.{RolePrincipalHistory, RolePrincipalHistoryId}
import ai.digital.deploy.permissions.repository.RolePrincipalHistoryRepository
import ai.digital.deploy.permissions.repository.jpa.RolePrincipalHistoryRepositoryImpl.{FindAll, FindByPrincipalNameQuery, FindByRoleIdQuery, FindByRolePrincipalQuery}
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.data.jpa.repository.{JpaRepository, Query}
import org.springframework.data.repository.query.Param

import java.time.LocalDateTime
import java.util.{List => JList, UUID}
import scala.jdk.CollectionConverters._

object RolePrincipalHistoryRepositoryImpl {
  final val FindByRolePrincipalQuery =
    "select h from RolePrincipalHistory h where h.rolePrincipalId = :rolePrincipalId and (h.startTime between :startTime and :endTime or h.endTime between :startTime and :endTime)"
  final val FindByPrincipalNameQuery =
    "select h from RolePrincipalHistory h where h.principalName = :principalName and (h.startTime between :startTime and :endTime or h.endTime between :startTime and :endTime)"
  final val FindByRoleIdQuery =
    "select h from RolePrincipalHistory h where h.roleId = :roleId and (h.startTime between :startTime and :endTime or h.endTime between :startTime and :endTime)"
  final val FindAll =
    "select h from RolePrincipalHistory h where (h.startTime between :startTime and :endTime or h.endTime between :startTime and :endTime)"
}

trait RolePrincipalHistoryRepositoryImpl
    extends RolePrincipalHistoryRepository
    with JpaRepository[RolePrincipalHistory, RolePrincipalHistoryId] {
  override def get(rolePrincipalId: UUID): List[RolePrincipalHistory] = findAllByRolePrincipalId(rolePrincipalId).asScala.toList

  override def get(principalName: String): List[RolePrincipalHistory] = findAllByPrincipalName(principalName).asScala.toList

  override def getRolePrincipalHistoriesForRole(roleId: UUID): List[RolePrincipalHistory] = findAllByRoleId(roleId).asScala.toList
  override def getRolePrincipalHistoriesForRoleBetween(roleId: UUID,
                                                       startTime: LocalDateTime,
                                                       endTime: LocalDateTime
  ): List[RolePrincipalHistory] = findAllByRoleIdAndForTimeRange(roleId, startTime, endTime).asScala.toList

  override def getRolePrincipalHistoryBetween(rolePrincipalId: UUID,
                                              startTime: LocalDateTime,
                                              endTime: LocalDateTime
  ): List[RolePrincipalHistory] = findAllByRolePrincipalIdAndForTimeRange(rolePrincipalId, startTime, endTime).asScala.toList

  override def getRolePrincipalHistoryBetween(principalName: String,
                                              startTime: LocalDateTime,
                                              endTime: LocalDateTime
  ): List[RolePrincipalHistory] = findAllByPrincipalNameAndForTimeRange(principalName, startTime, endTime).asScala.toList

  override def getRolePrincipalHistories(startTime: LocalDateTime, endTime: LocalDateTime): List[RolePrincipalHistory] =
    findAllByTimeRange(startTime, endTime).asScala.toList

  override def findAllRolePrincipalHistoriesForRole(roleId: UUID, pageable: Pageable): Page[RolePrincipalHistory] =
    findAllByRoleId(roleId, pageable)

  override def findAllRolePrincipalHistoriesForRoleBetween(roleId: UUID,
                                                           startTime: LocalDateTime,
                                                           endTime: LocalDateTime,
                                                           pageable: Pageable
  ): Page[RolePrincipalHistory] = findAllByRoleIdAndForTimeRange(roleId, startTime, endTime, pageable)

  override def findRolePrincipalHistories(rolePrincipalId: UUID,
                                          startTime: LocalDateTime,
                                          endTime: LocalDateTime,
                                          pageable: Pageable
  ): Page[RolePrincipalHistory] = findAllByRolePrincipalIdAndForTimeRange(rolePrincipalId, startTime, endTime, pageable)

  override def findRolePrincipalHistories(principalName: String,
                                          startTime: LocalDateTime,
                                          endTime: LocalDateTime,
                                          pageable: Pageable
  ): Page[RolePrincipalHistory] = findAllByPrincipalNameAndForTimeRange(principalName, startTime, endTime, pageable)

  override def findAllRolePrincipalHistories(startTime: LocalDateTime,
                                             endTime: LocalDateTime,
                                             pageable: Pageable
  ): Page[RolePrincipalHistory] = findAllByTimeRange(startTime, endTime, pageable)

  protected def findAllByRolePrincipalId(rolePrincipalId: UUID): JList[RolePrincipalHistory]

  protected def findAllByPrincipalName(principalName: String): JList[RolePrincipalHistory]

  protected def findAllByRoleId(roleId: UUID): JList[RolePrincipalHistory]

  protected def findAllByRoleId(roleId: UUID, pageable: Pageable): Page[RolePrincipalHistory]

  @Query(FindByRolePrincipalQuery)
  protected def findAllByRolePrincipalIdAndForTimeRange(@Param("rolePrincipalId") rolePrincipalId: UUID,
                                                        @Param("startTime") startTime: LocalDateTime,
                                                        @Param("endTime") endTime: LocalDateTime
  ): JList[RolePrincipalHistory]

  @Query(FindByRolePrincipalQuery)
  protected def findAllByRolePrincipalIdAndForTimeRange(@Param("rolePrincipalId") rolePrincipalId: UUID,
                                                        @Param("startTime") startTime: LocalDateTime,
                                                        @Param("endTime") endTime: LocalDateTime,
                                                        pageable: Pageable
  ): Page[RolePrincipalHistory]

  @Query(FindByPrincipalNameQuery)
  protected def findAllByPrincipalNameAndForTimeRange(@Param("principalName") principalName: String,
                                                      @Param("startTime") startTime: LocalDateTime,
                                                      @Param("endTime") endTime: LocalDateTime
  ): JList[RolePrincipalHistory]

  @Query(FindByPrincipalNameQuery)
  protected def findAllByPrincipalNameAndForTimeRange(@Param("principalName") principalName: String,
                                                      @Param("startTime") startTime: LocalDateTime,
                                                      @Param("endTime") endTime: LocalDateTime,
                                                      pageable: Pageable
  ): Page[RolePrincipalHistory]

  @Query(FindByRoleIdQuery)
  protected def findAllByRoleIdAndForTimeRange(@Param("roleId") roleId: UUID,
                                               @Param("startTime") startTime: LocalDateTime,
                                               @Param("endTime") endTime: LocalDateTime
  ): JList[RolePrincipalHistory]

  @Query(FindByRoleIdQuery)
  protected def findAllByRoleIdAndForTimeRange(@Param("roleId") roleId: UUID,
                                               @Param("startTime") startTime: LocalDateTime,
                                               @Param("endTime") endTime: LocalDateTime,
                                               pageable: Pageable
  ): Page[RolePrincipalHistory]

  @Query(FindAll)
  protected def findAllByTimeRange(@Param("startTime") startTime: LocalDateTime,
                                   @Param("endTime") endTime: LocalDateTime
  ): JList[RolePrincipalHistory]

  @Query(FindAll)
  protected def findAllByTimeRange(@Param("startTime") startTime: LocalDateTime,
                                   @Param("endTime") endTime: LocalDateTime,
                                   pageable: Pageable
  ): Page[RolePrincipalHistory]
}
