package ai.digital.deploy.permissions.configuration

import ai.digital.configuration.central.deploy.db.{DatabaseProperties, MainDatabase}
import ai.digital.configuration.central.deploy.permission.PermissionServiceProperties
import com.zaxxer.hikari.{HikariConfig, HikariDataSource}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.{Bean, Configuration}

import javax.sql.DataSource

@Configuration
class JdbcConfiguration(config: PermissionServiceProperties) {
  @Bean(destroyMethod = "close")
  def permissionDataSource(): DataSource = hikariDataSource()

  @Autowired(required = false)
  var mainDatabase: MainDatabase = _

  private def hikariDataSource() = new HikariDataSource(hikariConfig())

  private def hikariConfig() = {
    val dbConfig: DatabaseProperties = if (config.database.hasConfigured) config.database else mainDatabase.database
    val hikariConfig = new HikariConfig()
    var url = dbConfig.dbUrl
    // Microsoft SQL DB specific property sendStringParametersAsUnicode
    if (
      isMSSqlDatabase(dbConfig)
      && !url.contains("sendStringParametersAsUnicode")
    )
      url = url.concat(";sendStringParametersAsUnicode=false")
    hikariConfig.setDriverClassName(dbConfig.dbDriverClassname)
    hikariConfig.setJdbcUrl(url)
    hikariConfig.setUsername(dbConfig.dbUsername)
    hikariConfig.setPassword(dbConfig.dbPassword)
    hikariConfig.setMaximumPoolSize(dbConfig.maxPoolSize)
    hikariConfig.setMaxLifetime(dbConfig.getMaxLifeTimeInMillis)
    hikariConfig.setIdleTimeout(dbConfig.getIdleTimeoutInMillis)
    hikariConfig.setMinimumIdle(dbConfig.minimumIdle)
    hikariConfig.setConnectionTimeout(dbConfig.getConnectionTimeoutInMillis)
    hikariConfig.setTransactionIsolation("TRANSACTION_READ_COMMITTED")
    hikariConfig.setPoolName("PermissionPool")
    hikariConfig.setAutoCommit(false)
    hikariConfig.setLeakDetectionThreshold(dbConfig.getLeakDetectionThresholdInMillis)
    // check whats going on with hikariConfig.setMetricsTrackerFactory(metricsTrackerFactory)
    hikariConfig.setSchema(dbConfig.dbSchemaName)
    hikariConfig
  }

  def isMSSqlDatabase(dbConfig: DatabaseProperties): Boolean = dbConfig.dbDriverClassname.contains("microsoft")
}
