package ai.digital.deploy.permissions.repository.jpa

import ai.digital.deploy.permissions.model.view.{GlobalPermissionHistory, GlobalPermissionHistoryId}
import ai.digital.deploy.permissions.repository.GlobalPermissionHistoryRepository
import ai.digital.deploy.permissions.repository.jpa.GlobalPermissionHistoryRepositoryImpl.{FindAll, FindByGlobalPermissionIdQuery, FindByPermissionNameQuery, FindByRoleIdQuery}
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.data.jpa.repository.{JpaRepository, Query}
import org.springframework.data.repository.query.Param

import java.time.LocalDateTime
import java.util.{List => JList, UUID}
import scala.jdk.CollectionConverters._

object GlobalPermissionHistoryRepositoryImpl {
  final val FindByGlobalPermissionIdQuery =
    "select h from GlobalPermissionHistory h where h.globalPermissionId = :globalPermissionId and (h.startTime between :startTime and :endTime or h.endTime between :startTime and :endTime)"
  final val FindByPermissionNameQuery =
    "select h from GlobalPermissionHistory h where h.permissionName = :permissionName and (h.startTime between :startTime and :endTime or h.endTime between :startTime and :endTime)"
  final val FindByRoleIdQuery =
    "select h from GlobalPermissionHistory h where h.roleId = :roleId and (h.startTime between :startTime and :endTime or h.endTime between :startTime and :endTime)"
  final val FindAll =
    "select h from GlobalPermissionHistory h where (h.startTime between :startTime and :endTime or h.endTime between :startTime and :endTime)"
}

trait GlobalPermissionHistoryRepositoryImpl
    extends GlobalPermissionHistoryRepository
    with JpaRepository[GlobalPermissionHistory, GlobalPermissionHistoryId] {
  override def get(globalPermissionId: UUID): List[GlobalPermissionHistory] =
    findAllByGlobalPermissionId(globalPermissionId).asScala.toList
  override def get(permissionName: String): List[GlobalPermissionHistory] = findAllByPermissionName(permissionName).asScala.toList
  override def getGlobalPermissionHistoriesForRole(roleId: UUID): List[GlobalPermissionHistory] =
    findAllByRoleId(roleId).asScala.toList
  override def getGlobalPermissionHistoriesForRoleBetween(roleId: UUID,
                                                          startTime: LocalDateTime,
                                                          endTime: LocalDateTime
  ): List[GlobalPermissionHistory] = findAllByRoleIdForTimeRange(roleId, startTime, endTime).asScala.toList
  override def getGlobalPermissionHistoryBetween(globalPermissionId: UUID,
                                                 startTime: LocalDateTime,
                                                 endTime: LocalDateTime
  ): List[GlobalPermissionHistory] =
    findAllByGlobalPermissionIdForTimeRange(globalPermissionId, startTime, endTime).asScala.toList
  override def getGlobalPermissionHistoryBetween(permissionName: String,
                                                 startTime: LocalDateTime,
                                                 endTime: LocalDateTime
  ): List[GlobalPermissionHistory] = findAllByPermissionNameForTimeRange(permissionName, startTime, endTime).asScala.toList
  override def getGlobalPermissionHistories(startTime: LocalDateTime, endTime: LocalDateTime): List[GlobalPermissionHistory] =
    findAllByTimeRange(startTime, endTime).asScala.toList

  override def findAllGlobalPermissionHistoriesForRole(roleId: UUID, pageable: Pageable): Page[GlobalPermissionHistory] =
    findAllByRoleId(roleId, pageable)
  override def findAllGlobalPermissionHistoriesForRoleBetween(roleId: UUID,
                                                              startTime: LocalDateTime,
                                                              endTime: LocalDateTime,
                                                              pageable: Pageable
  ): Page[GlobalPermissionHistory] = findAllByRoleIdForTimeRange(roleId, startTime, endTime, pageable)
  override def findGlobalPermissionHistories(globalPermissionId: UUID,
                                             startTime: LocalDateTime,
                                             endTime: LocalDateTime,
                                             pageable: Pageable
  ): Page[GlobalPermissionHistory] = findAllByGlobalPermissionIdForTimeRange(globalPermissionId, startTime, endTime, pageable)
  override def findGlobalPermissionHistories(permissionName: String,
                                             startTime: LocalDateTime,
                                             endTime: LocalDateTime,
                                             pageable: Pageable
  ): Page[GlobalPermissionHistory] = findAllByPermissionNameForTimeRange(permissionName, startTime, endTime, pageable)
  override def findGlobalPermissionHistories(startTime: LocalDateTime,
                                             endTime: LocalDateTime,
                                             pageable: Pageable
  ): Page[GlobalPermissionHistory] = findAllByTimeRange(startTime, endTime, pageable)

  protected def findAllByGlobalPermissionId(globalPermissionId: UUID): JList[GlobalPermissionHistory]

  protected def findAllByPermissionName(permissionName: String): JList[GlobalPermissionHistory]

  protected def findAllByRoleId(roleId: UUID): JList[GlobalPermissionHistory]

  protected def findAllByRoleId(roleId: UUID, pageable: Pageable): Page[GlobalPermissionHistory]

  @Query(FindByRoleIdQuery)
  protected def findAllByRoleIdForTimeRange(@Param("roleId") roleId: UUID,
                                            @Param("startTime") startTime: LocalDateTime,
                                            @Param("endTime") endTime: LocalDateTime
  ): JList[GlobalPermissionHistory]
  @Query(FindByRoleIdQuery)
  protected def findAllByRoleIdForTimeRange(@Param("roleId") roleId: UUID,
                                            @Param("startTime") startTime: LocalDateTime,
                                            @Param("endTime") endTime: LocalDateTime,
                                            pageable: Pageable
  ): Page[GlobalPermissionHistory]
  @Query(FindByGlobalPermissionIdQuery)
  protected def findAllByGlobalPermissionIdForTimeRange(@Param("globalPermissionId") globalPermissionId: UUID,
                                                        @Param("startTime") startTime: LocalDateTime,
                                                        @Param("endTime") endTime: LocalDateTime
  ): JList[GlobalPermissionHistory]

  @Query(FindByGlobalPermissionIdQuery)
  protected def findAllByGlobalPermissionIdForTimeRange(@Param("globalPermissionId") globalPermissionId: UUID,
                                                        @Param("startTime") startTime: LocalDateTime,
                                                        @Param("endTime") endTime: LocalDateTime,
                                                        pageable: Pageable
  ): Page[GlobalPermissionHistory]

  @Query(FindByPermissionNameQuery)
  protected def findAllByPermissionNameForTimeRange(@Param("permissionName") permissionName: String,
                                                    @Param("startTime") startTime: LocalDateTime,
                                                    @Param("endTime") endTime: LocalDateTime
  ): JList[GlobalPermissionHistory]

  @Query(FindByPermissionNameQuery)
  protected def findAllByPermissionNameForTimeRange(@Param("permissionName") permissionName: String,
                                                    @Param("startTime") startTime: LocalDateTime,
                                                    @Param("endTime") endTime: LocalDateTime,
                                                    pageable: Pageable
  ): Page[GlobalPermissionHistory]

  @Query(FindAll)
  protected def findAllByTimeRange(@Param("startTime") startTime: LocalDateTime,
                                   @Param("endTime") endTime: LocalDateTime
  ): JList[GlobalPermissionHistory]

  @Query(FindAll)
  protected def findAllByTimeRange(@Param("startTime") startTime: LocalDateTime,
                                   @Param("endTime") endTime: LocalDateTime,
                                   pageable: Pageable
  ): Page[GlobalPermissionHistory]
}
