package ai.digital.deploy.permissions.repository.jpa

import ai.digital.deploy.permissions.jpa.TransactionalPermissionRepository
import ai.digital.deploy.permissions.model.{Role, RolePrincipal}
import ai.digital.deploy.permissions.repository.RolePrincipalRepository
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.data.jpa.repository.JpaRepository
import org.springframework.stereotype.Repository

import java.util.{List => JList, UUID}
import scala.jdk.CollectionConverters._

@Repository
@TransactionalPermissionRepository
trait RolePrincipalRepositoryImpl extends RolePrincipalRepository with JpaRepository[RolePrincipal, UUID] {
  override def get(): List[RolePrincipal] = findAll().asScala.toList
  override def get(principalName: String): List[RolePrincipal] = findRolesByPrincipalNameIgnoreCase(principalName).asScala.toList
  override def get(role: Role): List[RolePrincipal] = findByRole(role).asScala.toList
  override def getAll(principals: List[String]): List[RolePrincipal] =
    findRolesByPrincipalNameInIgnoreCase(principals.asJava).asScala.toList
  override def add(rolePrincipal: RolePrincipal): RolePrincipal = save(rolePrincipal)
  override def add(rolePrincipals: List[RolePrincipal]): List[RolePrincipal] = saveAll(rolePrincipals.asJava).asScala.toList
  override def list(pageable: Pageable): Page[RolePrincipal] = findAll(pageable)
  override def listByPrincipalNameAndRoleNamePattern(principalName: String,
                                                     roleNamePattern: String,
                                                     pageable: Pageable
  ): Page[RolePrincipal] =
    findRolesByPrincipalNameIgnoreCaseAndRoleNameContainingIgnoreCase(principalName, roleNamePattern, pageable)
  override def listByPrincipalName(principalName: String, pageable: Pageable): Page[RolePrincipal] =
    findRolesByPrincipalNameIgnoreCase(principalName, pageable)
  override def listByPrincipalNamesAndRoleNamePattern(principals: List[String],
                                                      roleNamePattern: String,
                                                      pageable: Pageable
  ): Page[RolePrincipal] =
    findRolesByPrincipalNameInIgnoreCaseAndRoleNameContainingIgnoreCase(principals.asJava, roleNamePattern, pageable)
  override def getByPrincipalsAndRolePattern(principals: List[String], rolePattern: String): List[RolePrincipal] =
    findRolesByPrincipalNameInIgnoreCaseAndRoleNameContainingIgnoreCase(principals.asJava, rolePattern).asScala.toList
  def getByPrincipals(principals: List[String]): List[RolePrincipal] =
    findRolesByPrincipalNameInIgnoreCase(principals.asJava).asScala.toList
  override def getByPrincipalAndRolePattern(principalName: String, roleNamePattern: String): List[RolePrincipal] =
    findRolesByPrincipalNameIgnoreCaseAndRoleNameContainingIgnoreCase(principalName, roleNamePattern).asScala.toList
  override def getByRolePattern(rolePattern: String): List[RolePrincipal] =
    findRolesByRoleNameContainingIgnoreCase(rolePattern).asScala.toList
  override def remove(referencedPermissions: RolePrincipal): Unit = delete(referencedPermissions)
  override def remove(rolePricipals: List[RolePrincipal]): Unit = deleteAll(rolePricipals.asJava)
  override def removeForRole(role: Role): Unit = deleteAllByRole(role)
  override def removeAll(): Unit = deleteAll()

  protected def findRolesByPrincipalNameIgnoreCase(principalName: String): JList[RolePrincipal]
  protected def findRolesByPrincipalNameIgnoreCase(principalName: String, pageable: Pageable): Page[RolePrincipal]
  protected def findRolesByPrincipalNameInIgnoreCase(principals: JList[String]): JList[RolePrincipal]
  protected def findRolesByPrincipalNameIgnoreCaseAndRoleNameContainingIgnoreCase(principalName: String,
                                                                                  roleName: String,
                                                                                  pageable: Pageable
  ): Page[RolePrincipal]
  def findRolesByPrincipalNameInIgnoreCaseAndRoleNameContainingIgnoreCase(principals: JList[String],
                                                                          roleNamePattern: String,
                                                                          pageable: Pageable
  ): Page[RolePrincipal]
  def findRolesByPrincipalNameInIgnoreCaseAndRoleNameContainingIgnoreCase(principals: JList[String],
                                                                          roleNamePattern: String
  ): JList[RolePrincipal]
  def findRolesByRoleNameContainingIgnoreCase(rolePattern: String): JList[RolePrincipal]
  protected def findRolesByPrincipalNameIgnoreCaseAndRoleNameContainingIgnoreCase(principalName: String,
                                                                                  roleName: String
  ): JList[RolePrincipal]
  protected def findByRole(role: Role): JList[RolePrincipal]
  protected def deleteAllByRole(role: Role): Unit
}
