package ai.digital.deploy.permissions.service.impl

import ai.digital.deploy.permissions.jpa.ReadOnlyTransactionalPermissionService
import ai.digital.deploy.permissions.model.view.{HistoryEntity, ReferencedPermissionHistory, RolePrincipalHistory}
import ai.digital.deploy.permissions.repository.{ReferencedPermissionHistoryRepository, RolePrincipalHistoryRepository}
import ai.digital.deploy.permissions.service.ReferencedPermissionHistoryService
import org.hibernate.envers.RevisionType
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.stereotype.Service

import java.time.LocalDateTime
import java.util.UUID

@Service
@ReadOnlyTransactionalPermissionService
class ReferencedPermissionHistoryServiceImpl(referencedPermissionHistoryRepository: ReferencedPermissionHistoryRepository,
                                             rolePrincipalHistoryRepository: RolePrincipalHistoryRepository
) extends ReferencedPermissionHistoryService {
  override def listReferencesByRoleAndPermissions(roleId: UUID,
                                                  permissions: List[String],
                                                  startTime: LocalDateTime,
                                                  endTime: LocalDateTime
  ): List[ReferencedPermissionHistory] =
    filterToMatchInterval(
      referencedPermissionHistoryRepository
        .getReferencedPermissionHistoriesForRoleAndPermissions(roleId, permissions),
      (e: ReferencedPermissionHistory) => e.referencedPermissionId,
      startTime,
      endTime
    )

  override def listReferencesByPrincipalAndPermissions(principalName: String,
                                                       permissions: List[String],
                                                       startTime: LocalDateTime,
                                                       endTime: LocalDateTime
  ): List[ReferencedPermissionHistory] = {
    val rolePrincipals = filterToMatchInterval(rolePrincipalHistoryRepository.get(principalName),
                                               (e: RolePrincipalHistory) => e.rolePrincipalId,
                                               startTime,
                                               endTime
    )
    rolePrincipals.flatMap(rolePrincipal =>
      filterToMatchInterval[ReferencedPermissionHistory](
        referencedPermissionHistoryRepository
          .getReferencedPermissionHistoriesForRoleAndPermissions(rolePrincipal.roleId, permissions),
        (e: ReferencedPermissionHistory) => e.referencedPermissionId,
        rolePrincipal.startTime,
        Option(rolePrincipal.endTime).getOrElse(endTime)
      )
    )
  }

  override def listReferencesByRoleAndReferenceAndPermissions(roleId: UUID,
                                                              reference: UUID,
                                                              permissions: List[String],
                                                              startTime: LocalDateTime,
                                                              endTime: LocalDateTime
  ): List[ReferencedPermissionHistory] =
    filterToMatchInterval(
      referencedPermissionHistoryRepository
        .getReferencedPermissionHistoriesForRoleAndReference(roleId, reference, permissions),
      (e: ReferencedPermissionHistory) => e.referencedPermissionId,
      startTime,
      endTime
    )

  override def listReferencesByPrincipalAndReferenceAndPermissions(principalName: String,
                                                                   reference: UUID,
                                                                   permissions: List[String],
                                                                   startTime: LocalDateTime,
                                                                   endTime: LocalDateTime
  ): List[ReferencedPermissionHistory] = {
    val rolePrincipals = filterToMatchInterval(rolePrincipalHistoryRepository.get(principalName),
                                               (e: RolePrincipalHistory) => e.rolePrincipalId,
                                               startTime,
                                               endTime
    )
    rolePrincipals.flatMap(rolePrincipal =>
      filterToMatchInterval(
        referencedPermissionHistoryRepository
          .getReferencedPermissionHistoriesForRoleAndReference(rolePrincipal.roleId, reference, permissions),
        (e: ReferencedPermissionHistory) => e.referencedPermissionId,
        rolePrincipal.startTime,
        Option(rolePrincipal.endTime).getOrElse(endTime)
      )
    )
  }

  private def filterToMatchInterval[T <: HistoryEntity](histories: List[T],
                                                        id: T => UUID,
                                                        startTime: LocalDateTime,
                                                        endTime: LocalDateTime
  ) =
    histories
      .groupBy(id).flatMap { case (_, groupedHistories: List[T]) =>
        filterGroupedHistoriesToMatchInterval(startTime, endTime, groupedHistories)
      }.toList

  private def filterGroupedHistoriesToMatchInterval[T <: HistoryEntity](startTime: LocalDateTime,
                                                                        endTime: LocalDateTime,
                                                                        groupedHistories: List[T]
  ) =
    groupedHistories match {
      case singletonGroupedHistories if singletonGroupedHistories.size == 1 =>
        // if we have single record and it falls before endTime then we take it
        singletonGroupedHistories.filter(rp => rp.startTime.isBefore(endTime))
      case multiGroupedHistories =>
        // take only "finished" (that is create or any possible modifies)
        // we ignore delete rev_types
        // we want to have those that at least one of the dates comes into range
        filterCreateAndModifyInTimeInterval(multiGroupedHistories, startTime, endTime)
    }

  private def filterCreateAndModifyInTimeInterval[T <: HistoryEntity](multiGroupedHistories: List[T],
                                                                      startTime: LocalDateTime,
                                                                      endTime: LocalDateTime
  ) =
    multiGroupedHistories.filter(rp =>
      rp.revisionType != Byte
        .byte2int(RevisionType.DEL.getRepresentation) && (
        (rp.startTime.isAfter(startTime) && rp.startTime.isBefore(endTime))
          || (rp.startTime.isBefore(startTime) && (rp.endTime == null || rp.endTime.isAfter(endTime)))
          || (rp.startTime
            .isBefore(endTime) && (rp.endTime == null || (rp.endTime.isAfter(startTime) && rp.endTime.isBefore(endTime))))
      )
    )

  override def getReferenceHistoriesForReferenceAndPermission(referenceId: UUID,
                                                              permission: String,
                                                              startTime: LocalDateTime,
                                                              endTime: LocalDateTime,
                                                              pageable: Pageable
  ): Page[ReferencedPermissionHistory] =
    referencedPermissionHistoryRepository.findReferencedPermissionHistories(permission, referenceId, startTime, endTime, pageable)

  override def getReferenceHistoriesForRoleId(roleId: UUID, pageable: Pageable): Page[ReferencedPermissionHistory] =
    referencedPermissionHistoryRepository.findAllReferencedPermissionHistoriesForRole(roleId, pageable)

  override def getReferenceHistoriesForReferenceAndRoleId(referenceId: UUID,
                                                          roleId: UUID,
                                                          startTime: LocalDateTime,
                                                          endTime: LocalDateTime,
                                                          pageable: Pageable
  ): Page[ReferencedPermissionHistory] =
    referencedPermissionHistoryRepository.findAllReferencedPermissionHistoriesForReferenceRoleBetween(roleId,
                                                                                                      referenceId,
                                                                                                      startTime,
                                                                                                      endTime,
                                                                                                      pageable
    )
  override def getReferenceHistoriesForReferenceAndReferencedPermissionId(referenceId: UUID,
                                                                          referencedPermissionId: UUID,
                                                                          startTime: LocalDateTime,
                                                                          endTime: LocalDateTime,
                                                                          pageable: Pageable
  ): Page[ReferencedPermissionHistory] =
    referencedPermissionHistoryRepository.findReferencedPermissionHistories(referencedPermissionId,
                                                                            referenceId,
                                                                            startTime,
                                                                            endTime,
                                                                            pageable
    )
}
