package ai.digital.deploy.permissions.service.impl

import ai.digital.deploy.permissions.jpa.{ReadOnlyTransactionalPermissionService, TransactionalPermissionService}
import ai.digital.deploy.permissions.model.view.RolePrincipalReferenceView
import ai.digital.deploy.permissions.model.{Role, RolePrincipal}
import ai.digital.deploy.permissions.repository.{RolePrincipalReferenceViewRepository, RolePrincipalRepository}
import ai.digital.deploy.permissions.service.{RoleNameNotFoundServiceException, RolePrincipalService, RoleService}
import org.springframework.data.domain.{Page, PageImpl, Pageable}
import org.springframework.stereotype.Service

import java.util.UUID
import scala.jdk.CollectionConverters._

@Service
@TransactionalPermissionService
class RolePrincipalServiceImpl(rolePrincipalRepository: RolePrincipalRepository,
                               rolePrincipalReferenceViewRepository: RolePrincipalReferenceViewRepository,
                               roleService: RoleService
) extends RolePrincipalService {
  override def add(roleName: String,
                   principals: List[String]
  ): Either[RoleNameNotFoundServiceException, (Role, List[RolePrincipal])] =
    roleService
      .read(roleName).map { role =>
        val (exists, toDelete) = rolePrincipalRepository.get(role).partition(rp => principals.contains(rp.principalName))
        val toSave = principals
          .filterNot { p =>
            exists.map(_.principalName).contains(p)
          }.map(p => RolePrincipal(role, p))
        rolePrincipalRepository.remove(toDelete)
        Right((role, exists ++ rolePrincipalRepository.add(toSave)))
      }.getOrElse(Left(RoleNameNotFoundServiceException(roleName)))

  override def remove(roleName: String,
                      principals: List[String]
  ): Either[RoleNameNotFoundServiceException, (Role, List[RolePrincipal])] =
    roleService
      .read(roleName).map { role =>
        val (toDelete, rp) = rolePrincipalRepository.get(role).partition(rp => principals.contains(rp.principalName))
        rolePrincipalRepository.remove(toDelete)
        Right((role, rp))
      }.getOrElse(Left(RoleNameNotFoundServiceException(roleName)))

  override def removeForRole(role: Role): Unit = rolePrincipalRepository.removeForRole(role)

  @ReadOnlyTransactionalPermissionService
  override def read(roleName: String): Either[RoleNameNotFoundServiceException, List[RolePrincipal]] =
    roleService
      .read(roleName).map { role =>
        Right(rolePrincipalRepository.get(role))
      }.getOrElse(Left(RoleNameNotFoundServiceException(roleName)))

  @ReadOnlyTransactionalPermissionService
  override def read(roleNamePattern: String, pageable: Pageable): (Page[Role], List[RolePrincipal]) = {
    val rolePage = roleService.read(roleNamePattern, pageable)
    val roleIds = rolePage.getContent.asScala.toList.map(_.id)
    val views = rolePrincipalRepository.get().filter(rp => roleIds.contains(rp.role.id))
    (rolePage, views)
  }

  @ReadOnlyTransactionalPermissionService
  override def read(principalName: String, roleNamePattern: String, pageable: Pageable): Page[RolePrincipal] =
    wrapNullablePatternSearch(
      roleNamePattern,
      p => rolePrincipalRepository.listByPrincipalNameAndRoleNamePattern(principalName, p, pageable),
      () => rolePrincipalRepository.listByPrincipalName(principalName, pageable)
    )

  @ReadOnlyTransactionalPermissionService
  override def read(referenceId: UUID,
                    roleNamePattern: String,
                    pageable: Pageable
  ): (Page[Role], List[RolePrincipalReferenceView]) = {
    val rolePage = roleService.read(roleNamePattern, pageable)
    val roleIds = rolePage.getContent.asScala.toList.map(_.id)
    val views = rolePrincipalReferenceViewRepository.get(roleIds, referenceId)
    val roleIdsOnReference = views.map(_.roleId).distinct
    val rolesOnReference = rolePage.getContent.asScala.toList.filter(role => roleIdsOnReference.contains(role.id))
    val rolePageForReference = new PageImpl[Role](rolesOnReference.asJava, pageable, rolesOnReference.size)
    (rolePageForReference, views)
  }

  @ReadOnlyTransactionalPermissionService
  override def readAll(principals: List[String]): List[RolePrincipal] = rolePrincipalRepository.getAll(principals)

  override def edit(roleName: String,
                    principalsToAdd: List[String],
                    principalsToDelete: List[String]
  ): Either[RoleNameNotFoundServiceException, (Role, List[RolePrincipal])] =
    roleService
      .read(roleName).map { role =>
        val (toDelete, exists) = rolePrincipalRepository.get(role).partition(p => principalsToDelete.contains(p.principalName))
        rolePrincipalRepository.remove(toDelete)
        val toAdd = principalsToAdd.map(p => RolePrincipal(role, p))
        Right((role, exists ++ rolePrincipalRepository.add(toAdd)))
      }.getOrElse(Left(RoleNameNotFoundServiceException(roleName)))

  @ReadOnlyTransactionalPermissionService
  override def read(principals: List[String], roleNamePattern: String, pageable: Pageable): Page[RolePrincipal] =
    rolePrincipalRepository.listByPrincipalNamesAndRoleNamePattern(principals, roleNamePattern, pageable)

  @ReadOnlyTransactionalPermissionService
  override def readByRolePattern(rolePattern: String): (List[Role], List[RolePrincipal]) = {
    val roles = roleService.readByRolePattern(rolePattern)
    val roleIds = roles.map(_.id)
    val views = rolePrincipalRepository.get().filter(rp => roleIds.contains(rp.role.id))
    (roles, views)
  }

  @ReadOnlyTransactionalPermissionService
  override def read(principal: String, rolePattern: String): List[RolePrincipal] =
    wrapNullablePatternSearch(
      rolePattern,
      p => rolePrincipalRepository.getByPrincipalAndRolePattern(principal, p),
      () => rolePrincipalRepository.get(principal)
    )

  @ReadOnlyTransactionalPermissionService
  override def read(principals: List[String], rolePattern: String): List[RolePrincipal] =
    wrapNullablePatternSearch(
      rolePattern,
      p => rolePrincipalRepository.getByPrincipalsAndRolePattern(principals, p),
      () => rolePrincipalRepository.getAll(principals)
    )

  @ReadOnlyTransactionalPermissionService
  override def readByPrincipal(principalName: String): List[RolePrincipal] = rolePrincipalRepository.get(principalName)
}
