package com.xebialabs.deployit.plugin.cmd.step;

import com.xebialabs.deployit.local.message.ProductName;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.cmd.deployed.DeployedCommand;
import com.xebialabs.deployit.plugin.cmd.whitelist.validator.CommandWhitelistValidator;
import com.xebialabs.deployit.plugin.cmd.whitelist.validator.CommandWhitelistValidatorHolder;
import com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler;
import com.xebialabs.overthere.*;
import com.xebialabs.xlplatform.satellite.Satellite;
import com.xebialabs.xlplatform.satellite.SatelliteAware;

import java.io.IOException;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.google.common.base.Splitter.on;
import static com.google.common.base.Strings.nullToEmpty;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStderr;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStdout;

@SuppressWarnings("serial")
public class ExecuteCommandStep implements SatelliteAware {

    private final DeployedCommand command;
    private final Pattern placeHolderPattern = Pattern.compile("\\$\\{(.*)}");
    private final int order;

    private transient OverthereConnection localConn;
    private transient OverthereConnection remoteConn;
    private transient ExecutionContext ctx;


    public ExecuteCommandStep(int order, DeployedCommand deployedCommand) {
        this.order = order;
        this.command = deployedCommand;
        validateCommandLineDependencyPlaceholders();
    }

    /*
     * Cannot store the commandLineArgs Iterable as a field, as it is not serializable..
     * (See: DEPLOYITPB-2234)
     */
    private Iterable<String> getCommandLineArgs() {
        String params = command.getCommandLine();
        if (!nullToEmpty(params).trim().isEmpty()) {
            params = params.replace('\n', ' ');
            return on(' ').split(params);
        }
        return newArrayList();
    }

    private void validateCommandLineDependencyPlaceholders() {
        for (String arg : getCommandLineArgs()) {
            String dependencyRef = extractDependencyRefFromPlaceholderArg(arg);
            if (dependencyRef != null) {
                boolean found = false;
                for (DeployableArtifact dependency : command.getDependencies()) {
                    if (dependency.getName().equals(dependencyRef)) {
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    throw new IllegalArgumentException("Placeholder arg " + arg + " does not refer to a valid dependency");
                }
            }
        }
    }

    private String extractDependencyRefFromPlaceholderArg(String arg) {
        Matcher matcher = placeHolderPattern.matcher(arg);
        if (matcher.matches()) {
            return matcher.group(1);
        }
        return null;
    }

    @Override
    public String getDescription() {
        return "Execute " + command.getName();
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) {
        try {
            this.ctx = ctx;
            Map<String, OverthereFile> uploadedDependencies = uploadDependencies();
            CmdLine cmdLine = resolveCommandLine(uploadedDependencies);
            try (DefaultExecutionOutputHandler stdoutHandler = handleStdout(ctx);
                 DefaultExecutionOutputHandler stderrHandler = handleStderr(ctx)) {
                String cmdString = cmdLine.toCommandLine(command.getContainer().getOs(), true);

                commandWhiteListValidation(ctx, cmdString);
                ctx.logMsgOutput(ProductName.DEPLOY, "command.executing.line", cmdString);
                int rc = getRemoteConnection().execute(stdoutHandler, stderrHandler, cmdLine);
                if (rc != 0) {
                    ctx.logError("Command failed with return code " + rc);
                    return StepExitCode.FAIL;
                }
            } catch (RuntimeException | IOException ex) {
                ctx.logError(ex.getMessage());
                return StepExitCode.FAIL;
            }
        } finally {
            disconnect();
        }

        return StepExitCode.SUCCESS;
    }

    private void commandWhiteListValidation(final ExecutionContext ctx, final String cmdString) throws IOException {
        Optional<CommandWhitelistValidator> validator = CommandWhitelistValidatorHolder.getValidatorInstance();
        if (validator.isPresent()) {
            validator.get().validate(ctx.getTask().getUsername(), cmdString);
        } else {
            ctx.logOutput("Skipping command validation, continuing with deployment");
        }
    }

    private Map<String, OverthereFile> uploadDependencies() {
        Map<String, OverthereFile> uploadedFiles = newHashMap();
        if (command.getDependencies() == null || command.getDependencies().isEmpty()) {
            return uploadedFiles;
        }

        OverthereFile tempDir = getRemoteConnection().getTempFile("exec_cmd", ".tmp");
        tempDir.mkdir();
        ctx.logOutput("Uploading " + command.getDependencies().size() + " dependent files to " + tempDir.getPath());
        for (DeployableArtifact artifact : command.getDependencies()) {
            String name = artifact.getName();
            OverthereFile uploadedFile = tempDir.getFile(name);
            OverthereFile localFile = getLocalConnection().getFile(artifact.getFile().getPath());
            localFile.copyTo(uploadedFile);
            ctx.logOutput("Uploaded " + name);
            uploadedFiles.put(name, uploadedFile);
        }
        ctx.logOutput("Uploading done.");
        return uploadedFiles;
    }

    private CmdLine resolveCommandLine(Map<String, OverthereFile> dependencies) {
        CmdLine cmdLine = new CmdLine();
        for (String arg : getCommandLineArgs()) {
            String dependencyRef = extractDependencyRefFromPlaceholderArg(arg);
            if (dependencyRef != null) {
                cmdLine.addArgument(dependencies.get(dependencyRef).getPath());
            } else {
                cmdLine.addArgument(arg);
            }
        }
        return cmdLine;
    }

    private OverthereConnection getLocalConnection() {
        if (localConn == null) {
            localConn = Overthere.getConnection("local", new ConnectionOptions());
        }
        return localConn;
    }

    private OverthereConnection getRemoteConnection() {
        if (remoteConn == null) {
            remoteConn = command.getContainer().getConnection();
        }
        return remoteConn;
    }

    private void disconnect() {
        if (localConn != null)
            localConn.close();

        if (remoteConn != null)
            remoteConn.close();

        localConn = null;
        remoteConn = null;
    }

    @Override
    public int getOrder() {
        return order;
    }

    @Override
    public Satellite getSatellite() {
        return command.getContainer().getHost().getSatellite();
    }
}
