package com.xebialabs.deployit.plugin.cmd.whitelist.validator;

import ai.digital.configuration.central.deploy.CommandWhitelistProperties;
import ai.digital.configuration.central.deploy.Roles;
import com.xebialabs.deployit.engine.tasker.query.QueryExecutor;
import com.xebialabs.deployit.engine.tasker.query.QueryExecutorHandler;
import com.xebialabs.deployit.plugin.cmd.whitelist.exception.CommandWhitelistException;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import org.springframework.security.core.context.SecurityContextHolder;
import scala.util.Left;
import scala.util.Right;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

public class LocalCommandWhitelistValidator implements CommandWhitelistValidator, QueryExecutor {

    private final CommandWhitelistProperties commandWhitelistProperties;
    private final RoleService roleService;

    public LocalCommandWhitelistValidator(CommandWhitelistProperties commandWhitelistProperties, RoleService roleService) {
        this.commandWhitelistProperties = commandWhitelistProperties;
        this.roleService = roleService;
    }

    public void validate(String user, String cmd) {

        if (commandWhitelistProperties == null)
            return;

        if (commandWhitelistProperties.enabled()) {
            List<String> allAllowed = new ArrayList<>(commandWhitelistProperties.allUsers().allowedCommands());
            List<String> allRestricted = new ArrayList<>(commandWhitelistProperties.allUsers().restrictedCommands());
            validateCommandsForAllUsers(allAllowed, allRestricted, cmd);

            List<Role> userRoles = roleService.getRolesFor(user);

            //INFO: handling external users
            if(userRoles.isEmpty()){
                userRoles = roleService.getRolesFor(SecurityContextHolder.getContext().getAuthentication());
            }
            for (Role userRole : userRoles) {
                if (commandWhitelistProperties.roles().stream().anyMatch(role -> role.roleName().equals(userRole.getName()))) {
                    Roles roleRule = commandWhitelistProperties.roles().stream()
                            .filter(role -> role.roleName().equals(userRole.getName()))
                            .findFirst()
                            .orElse(new Roles());
                    validateCommandsForCurrentUser(roleRule, cmd);
                    allAllowed.addAll(roleRule.allowedCommands());
                    allRestricted.addAll(roleRule.restrictedCommands());
                }
            }

            validateWhitelisted(allAllowed, allRestricted, cmd);
        }
    }

    @Override
    public Serializable query(final QueryExecutorHandler.QueryPayload payload) {
        CommandWhitelistPayload commandWhitelistPayload = (CommandWhitelistPayload)payload;
        try {
            validate(commandWhitelistPayload.getUser(), commandWhitelistPayload.getCmd());
            return Right.apply(true);
        } catch (Exception e) {
            return Left.apply(e);
        }
    }

    private void validateWhitelisted(List<String> allAllowed, List<String> allRestricted, String cmd) {
        if (!allAllowed.isEmpty() && !allRestricted.isEmpty()) {
            throw new CommandWhitelistException("Both allowed commands and restricted commands specified. Please set only one of those properties.");
        }

        if (!allAllowed.isEmpty() && (allAllowed.stream().noneMatch(regex -> Pattern.matches(regex, cmd)))) {
            throw new CommandWhitelistException("Command " + cmd + " is not whitelisted in allowed-commands.");
        }
    }

    private void validateCommandsForCurrentUser(Roles roleRule, String cmd) {
        String roleName = roleRule.roleName();
        List<String> allowed = roleRule.allowedCommands();
        List<String> restricted = roleRule.restrictedCommands();

        if (!allowed.isEmpty() && !restricted.isEmpty()) {
            throw new CommandWhitelistException("Both allowed commands and restricted commands specified for role " + roleName + ". Please set only one of those properties.");
        }

        if (!restricted.isEmpty()) {
            restricted.forEach( regex -> {
                if (Pattern.matches(regex, cmd)) {
                    throw new CommandWhitelistException("Command " + cmd + " is restricted for role " + roleName + " by rule " + regex);
                }
            });
        }
    }

    private void validateCommandsForAllUsers(List<String> allowed, List<String> restricted, String cmd) {
        if (!allowed.isEmpty() && !restricted.isEmpty()) {
            throw new CommandWhitelistException("Both allowed commands and restricted commands specified for all users. Please set only one of those properties.");
        }

        if (!restricted.isEmpty()) {
            restricted.forEach( regex -> {
                if (Pattern.matches(regex, cmd)) {
                    throw new CommandWhitelistException("Command " + cmd + " is restricted by rule " + regex + " for all users.");
                }
            });
        }
    }
}
