package com.xebialabs.deployit.plugins.releaseauth;

import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableSet.Builder;

import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.DeploymentPackage;
import com.xebialabs.deployit.plugin.api.udm.Version;

import static com.google.common.base.Strings.nullToEmpty;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.BOOLEAN;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;
import static java.lang.Boolean.TRUE;
import static java.lang.String.format;

public final class ConditionVerifier {

    private ConditionVerifier() {}

    public static VerificationResult validateReleaseConditions(Set<ReleaseCondition> conditions, Version deploymentPackage) {
        VerificationResult result = new VerificationResult();

        for (ReleaseCondition condition : conditions) {
            if (!deploymentPackage.hasProperty(condition.getName())) {
                String environmentConditionName = condition.getName().replaceFirst("satisfies", "requires");
                throw new RuntimeException(
                        format("Property '%s' is missing from the definition of udm.DeploymentPackage. It is required since udm.Environment has a corresponding property named '%s'",
                                condition.getName(), environmentConditionName));
            }

            PropertyKind kind = Type.valueOf(DeploymentPackage.class).getDescriptor().getPropertyDescriptor(condition.getName()).getKind();
            Object value = deploymentPackage.getProperty(condition.getName());
            logger.debug("condition value for condition {} is {}", condition.getName(), value);
            switch (kind) {
            case BOOLEAN:
                verifyBooleanCondition(condition.getLabel(), (Boolean) value, result);
                break;
            case STRING:
                verifyStringCondition(condition.getLabel(), (String) value, result);
                break;
            default:
                throw new IllegalArgumentException(format("Only release conditions of kind '%s' or '%s' are supported, but condition '%s' was of kind '%s'",
                        BOOLEAN, STRING, condition.getName(), kind));
            }
        }
        return result;
    }

    private static void verifyBooleanCondition(String conditionName, Boolean conditionValue, VerificationResult result) {
        if (!TRUE.equals(conditionValue)) {
            result.logViolatedCondition(conditionName, TRUE, conditionValue);
        } else {
            result.logValidatedCondition(conditionName);
        }
    }

    private static void verifyStringCondition(String conditionName, String conditionValue, VerificationResult result) {
        if (isNullOrBlank(conditionValue)) {
            result.logViolatedCondition(conditionName, "non-empty value", conditionValue);
        } else {
            result.logValidatedCondition(conditionName);
        }
    }

    // TODO: move to Strings2 or whatever
    private static boolean isNullOrBlank(String string) {
        return nullToEmpty(string).trim().isEmpty();
    }

    public static class VerificationResult {
        private final ImmutableSet.Builder<ViolatedCondition<?>> violatedConditions = new Builder<>();
        private final ImmutableSet.Builder<String> validatedConditions = new Builder<>();
        private boolean failed = false;

        private <T> void logViolatedCondition(String condition, T expectedValue, T actualValue) {
            violatedConditions.add(new ViolatedCondition<T>(condition, expectedValue, actualValue));
            failed = true;
        }

        private void logValidatedCondition(String condition) {
            validatedConditions.add(condition);
        }

        public boolean failed() {
            return failed;
        }

        public Set<ViolatedCondition<?>> getViolatedConditions() {
            return violatedConditions.build();
        }

        public Set<String> getValidatedConditions() {
            return validatedConditions.build();
        }
    }

    public static class ViolatedCondition<T> {
        public final String name;
        public final T expectedValue;
        public final T actualValue;

        private ViolatedCondition(String name, T expectedValue, T actualValue) {
            this.name = name;
            this.expectedValue = expectedValue;
            this.actualValue = actualValue;
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(ConditionVerifier.class);
}
