package com.xebialabs.deployit.plugin.file.deployed;

import com.xebialabs.deployit.plugin.api.deployment.planning.*;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployedArtifact;
import com.xebialabs.deployit.plugin.file.FileCopyStrategy;
import com.xebialabs.deployit.plugin.generic.ci.Container;
import com.xebialabs.deployit.plugin.generic.deployed.CopiedArtifact;
import com.xebialabs.deployit.plugin.overthere.DefaultCopyStrategyName;
import com.xebialabs.deployit.plugin.overthere.Host;

import java.util.Set;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "Abstract deployed that can target any DeployableArtifact to a Host.")
public class DeployedArtifactOnHost<D extends DeployableArtifact> extends BaseDeployedArtifact<D, Host>  {

    @Property(description = "Path to which artifact must be copied to on the host.")
    private String targetPath;

    @Property(required = false, defaultValue = "true", description = "Is the targetPath shared by others on the host. When true, the targetPath is not deleted during undeployment; only the artifacts copied to it.")
    private boolean targetPathShared;

    @Property(required = false, defaultValue = "false", description = "Create the targetPath on the host if it does not exist.")
    private boolean createTargetPath;

    @Property(required = false, description = "Name of the artifact on the host.")
    private String targetFileName;

    @Property(defaultValue = "false", required = false, hidden = true, description = "If set to true, files are directly copied to the target directory without first being uploaded to the staging or temporary directory (like versions of XL Deploy prior to 4.0.0)")
    private boolean copyDirectlyToTargetPath;

    @Property(defaultValue = "false", required = false, hidden = true, description = "The sub directories on the target machine are not deleted if files other than that copied by XL Deploy are present. ")
    private boolean targetPathSharedSubDirectories;

    @Property(
            required = false,
            label = "Default Copy Strategy",
            description = "Default copy strategy for this folder artifact."
    )
    private FileCopyStrategy defaultCopyStrategy;

    public FileCopyStrategy getDefaultCopyStrategy() {
        return defaultCopyStrategy;
    }

    public void setDefaultCopyStrategy(FileCopyStrategy defaultCopyStrategy) {
        this.defaultCopyStrategy = defaultCopyStrategy;
    }

    public DefaultCopyStrategyName getDefaultCopyStrategyName() {
        FileCopyStrategy defaultCopyStrategy = getDefaultCopyStrategy();
        return defaultCopyStrategy !=null ? defaultCopyStrategy.getDefaultCopyStrategy() : null;
    }

    private CopiedArtifact<D> createDelegate() {
        // Not using the descriptor (which you should always do), because CopiedArtifact is virtual.
        // TODO maybe make a synthetic type which can be used here, which extends CopiedArtifact, just to be used as a delegate.
        CopiedArtifact<D> delegate = new CopiedArtifact<>(this.getId());
        Descriptor thisDescriptor = getType().getDescriptor();
        delegate.getType().getDescriptor().getPropertyDescriptors().stream()
                .filter(pd -> thisDescriptor.getPropertyDescriptor(pd.getName()) != null)
                .forEach(pd -> pd.set(delegate, thisDescriptor.getPropertyDescriptor(pd.getName()).get(this)));
        delegate.setPlaceholders(this.getPlaceholders());
        delegate.setFile(this.getFile());

        // Not using the descriptor (which you should always do), because Container is virtual.
        // TODO maybe make a synthetic type which can be used here, which extends Container, just to be used as a delegate.
        Container container = new Container(getContainer().getId());
        container.setHost(getContainer());

        delegate.setContainer(container);
        // Resolve the "members" property and inject it into the DeployableArtifact
        Set<String> resolvedMembers = (Set<String>) syntheticProperties.get("members");
        D deployable = this.getDeployable();
        if (resolvedMembers != null) {
            deployable.setProperty("members", resolvedMembers);
        }
        String stripComponents = (String) syntheticProperties.get("stripComponents");
        if (stripComponents != null) {
            deployable.setProperty("stripComponents", stripComponents);
        }
        delegate.setDeployable(deployable);
        delegate.setCreateOrder(DefaultOrders.CREATE_RESOURCES);
        delegate.setDestroyOrder(DefaultOrders.DESTROY_RESOURCES);

        delegate.setCreateTargetDirectory(isCreateTargetPath());
        delegate.setTargetFile(getTargetFileName());
        delegate.setTargetDirectory(getTargetPath());
        delegate.setTargetDirectoryShared(isTargetPathShared());
        delegate.setUseDescriptionGeneratedByStep(true);
        delegate.setCopyDirectlyToTargetPath(copyDirectlyToTargetPath);
        delegate.setTargetPathSharedSubDirectories(targetPathSharedSubDirectories);

        return delegate;
    }

    @Create
    public void executeCreate(DeploymentPlanningContext ctx, Delta delta) {
        CopiedArtifact<D> delegate = createDelegate();
        delegate.executeCreate(ctx, delta);
    }

    @SuppressWarnings("unchecked")
    @Modify
    public void executeModify(DeploymentPlanningContext ctx, Delta delta) {
        DeployedArtifactOnHost<D> previous = (DeployedArtifactOnHost<D>) delta.getPrevious();
        previous.executeDestroy(ctx, delta);
        executeCreate(ctx, delta);
    }

    @Destroy
    public void executeDestroy(DeploymentPlanningContext ctx, Delta delta) {
        CopiedArtifact<D> delegate = createDelegate();
        delegate.executeDestroy(ctx, delta);
    }

    public String getTargetPath() {
        return targetPath;
    }

    public void setTargetPath(String targetPath) {
        this.targetPath = targetPath;
    }

    public boolean isTargetPathShared() {
        return targetPathShared;
    }

    public void setTargetPathShared(boolean targetPathShared) {
        this.targetPathShared = targetPathShared;
    }

    public boolean isCreateTargetPath() {
        return createTargetPath;
    }

    public void setCreateTargetPath(boolean createTargetPath) {
        this.createTargetPath = createTargetPath;
    }

    public String getTargetFileName() {
        return targetFileName;
    }

    public void setTargetFileName(String targetFileName) {
        this.targetFileName = targetFileName;
    }

    public boolean isCopyDirectlyToTargetPath() {
        return copyDirectlyToTargetPath;
    }

    public void setCopyDirectlyToTargetPath(final boolean copyDirectlyToTargetPath) {
        this.copyDirectlyToTargetPath = copyDirectlyToTargetPath;
    }

    public boolean isTargetPathSharedSubDirectories() {
        return targetPathSharedSubDirectories;
    }

    public void setTargetPathSharedSubDirectories(final boolean targetPathSharedSubDirectories) {
        this.targetPathSharedSubDirectories = targetPathSharedSubDirectories;
    }
}
