package com.xebialabs.deployit.plugin.generic.step;

import com.xebialabs.deployit.plugin.api.Deprecations;
import com.xebialabs.deployit.plugin.api.flow.StageableStep;
import com.xebialabs.deployit.plugin.api.flow.StagedFile;
import com.xebialabs.deployit.plugin.api.flow.StagingContext;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployableFolderArtifact;
import com.xebialabs.deployit.plugin.generic.deployed.CopiedArtifact;
import com.xebialabs.deployit.plugin.overthere.CopyStrategyName;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.overthere.ConnectionOptions;
import com.xebialabs.overthere.OverthereFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.Objects;

import static com.xebialabs.deployit.io.ArtifactFile.ARTIFACTFILE_COPYSTRATEGY_KEY;


@SuppressWarnings("serial")
public class ArtifactCopyStep extends BaseFolderDeploymentStepSupport implements StageableStep {

    private static final Logger logger =  LoggerFactory.getLogger(ArtifactCopyStep.class);

    // cf. synthetic.xml in the remoting-plugin
    public static final String COPYSTRATEGY_PROPERTY = "copyStrategy";
    private static final String FILE_FOLDER_TYPE_PROPERTY="file.Folder";
    private OverthereFile sourceFile;

    //This parameter makes sense only for direct java usage. No need to annotate with @StepParameter
    private String sourceFileDescription;

    private Artifact sourceArtifact;

    private String targetPath;

    private boolean createTargetPath;

    private String targetFileName;

    //Weird logic is behind this field. Do not expose as @StepParameter
    private boolean preserveExistingFiles;

    private StagedFile stagedSourceArtifact;

    public ArtifactCopyStep(int order, OverthereFile sourceFile, HostContainer container, String targetPath) {
        super(order, container);
        this.sourceFile = sourceFile;
        this.targetPath = targetPath;
        Objects.requireNonNull(targetPath);
    }

    @Override
    public void requestStaging(final StagingContext ctx) {
        if (sourceArtifact != null) {
            logger.debug("[{}] is requesting staging of [{}]", this, sourceArtifact);
            stagedSourceArtifact = ctx.stageArtifact(sourceArtifact, container.getHost());
        } else {
            logger.debug("[{}] is not requesting artifact staging.", this);
        }
    }

    @Override
    protected StepExitCode doExecute() throws Exception {
        Map<String, String> metadata = getCtx().getTask().getMetadata();
        try {
            if (stagedSourceArtifact != null) {
                setRemoteCopyOptions();
                return doCopy(stagedSourceArtifact.get(getRemoteConnection(), getCtx()));

            } else {
                return doCopy(resolveSourceFile());
            }
        } catch (Exception e) {
            if (Boolean.parseBoolean(metadata.get("enableCopyArtifactRetry"))) {
                getCtx().logErrorRaw(e.getMessage() + "\n");
                return StepExitCode.RETRY;
            }
            throw e;
        }

    }

    @Override
    BaseDeployableFolderArtifact getBaseDeployableFolderArtifact() {
        BaseDeployableFolderArtifact deployableFolderArtifact = null;
        if (getSourceArtifact() != null) {
            if (this.getSourceArtifact() instanceof CopiedArtifact<?> ca && ca.getDeployable() != null && ca.getDeployable() instanceof BaseDeployableFolderArtifact deployable) {
                deployableFolderArtifact =  deployable;
            } else {
                logger.debug("Source artifact {} is not a CopiedArtifact or does not have a deployable of type BaseDeployableFolderArtifact", getSourceArtifact().getId());
            }
        } else {
            logger.debug("Source artifact is null, cannot resolve BaseDeployableFolderArtifact");
        }
        return deployableFolderArtifact;
    }

    private void setRemoteCopyOptions() {
        ConnectionOptions connOptions = getRemoteConnection().getOptions();
        Host host = container.getHost();
        if (connOptions.getOptional(ARTIFACTFILE_COPYSTRATEGY_KEY) != null || !host.hasProperty(COPYSTRATEGY_PROPERTY))
            return;

        CopyStrategyName strategy = host.getProperty(COPYSTRATEGY_PROPERTY);
        logger.debug("Setting artifact file copy strategy to {} for host {}", strategy, host.getId());
        connOptions.set(ARTIFACTFILE_COPYSTRATEGY_KEY, strategy);
    }

    private StepExitCode doCopy(OverthereFile sourceFile) {
        Objects.requireNonNull(targetFileName);
        OverthereFile remoteDir = getRemoteConnection().getFile(targetPath);
        if (!remoteDir.exists()) {
            if (createTargetPath) {
                getCtx().logOutput("Creating path " + targetPath + " on host " + getContainer().getHost());
                remoteDir.mkdirs();
            } else if (sourceFile.isFile()) {
                getCtx().logError("Path " + targetPath + " on host " + getContainer().getHost() + " does not exist.");
                return StepExitCode.FAIL;
            }
        }
        copyArtifact(sourceFile, remoteDir);
        return StepExitCode.SUCCESS;
    }

    private void copyArtifact(OverthereFile sourceFile, OverthereFile remoteDir) {
        BaseDeployableFolderArtifact baseDeployableFolderArtifact = getBaseDeployableFolderArtifact();
        if(baseDeployableFolderArtifact != null) {
            Artifact sa = this.getSourceArtifact();
            //TODO we have to load the allowed sa type from configuration files to have flexibility in folder plugin
            if (isForceArchivedForFolder() && sa != null && FILE_FOLDER_TYPE_PROPERTY.equals(baseDeployableFolderArtifact.getType().toString())) {
                if(isOneByOneCopyStrategy()) {
                    logger.debug("Artifact:Copy:Source artifact {} is a Folder artifact, proceeding with copy one by one", sa.getId());
                    getCtx().logOutput("Target directory:" + remoteDir.getPath() + " proceeding with copy from tmp likely one by one strategy will be very slow");
                    sourceFile.copyTo(getRemoteFile(sourceFile, remoteDir));
                } else {
                    logger.debug("Artifact:Copy:Source artifact {} is a Folder artifact, force archive copied", sa.getId());
                    getCtx().logOutput("Target directory:" + remoteDir.getPath() + " already artifact copied, so copy step skipped.");
                }
            }
            else {
                logger.debug("Artifact:Copy:Source artifact {} is not a Folder artifact, proceeding with copy", sa != null ? sa.getId() : "null");
                getCtx().logOutput("Target directory:" + remoteDir.getPath() + " proceeding with copy from tmp");
                sourceFile.copyTo(getRemoteFile(sourceFile, remoteDir));
            }
        } else {
            logger.debug("Artifact:Copy:Source file {} is not a Folder artifact, proceeding with copy", sourceFile.getPath());
            getCtx().logOutput("Target directory:" + remoteDir.getPath() + " proceeding with copy from tmp");
            sourceFile.copyTo(getRemoteFile(sourceFile, remoteDir));
        }
    }

    private OverthereFile getRemoteFile(OverthereFile sourceFile, OverthereFile remoteDir) {
        OverthereFile remoteFile;
        if (sourceFile.isDirectory()) {
            remoteFile = remoteDir;
        } else {
            remoteFile = remoteDir.getFile(getTargetFileName());
            if (remoteFile.exists()) {
                getCtx().logOutput(remoteFile.getPath() + " already exists on host " + getContainer().getHost() + ". Will replace.");
                remoteFile.delete();
            }
        }
        return remoteFile;
    }

    protected OverthereFile resolveSourceFile() {
        Objects.requireNonNull(sourceFile);
        return sourceFile;
    }

    @Override
    public String getDescription() {
        String description = super.getDescription();
        if (description == null) {
            return "Copy " + getSourceFileDescription() + " to " + getContainer().getHost();
        }
        return description;
    }

    public String getSourceFileDescription() {
        return sourceFileDescription;
    }

    public void setSourceFileDescription(final String sourceFileDescription) {
        this.sourceFileDescription = sourceFileDescription;
    }

    public Artifact getSourceArtifact() {
        return sourceArtifact;
    }

    public void setSourceArtifact(final Artifact sourceArtifact) {
        this.sourceArtifact = sourceArtifact;
    }

    public boolean isCreateTargetPath() {
        return createTargetPath;
    }

    public void setCreateTargetPath(final boolean createTargetPath) {
        this.createTargetPath = createTargetPath;
    }

    public String getTargetFileName() {
        return targetFileName;
    }

    public String getTargetPath() {
        return targetPath;
    }

    public void setTargetFileName(final String targetFileName) {
        this.targetFileName = targetFileName;
    }

    public boolean isPreserveExistingFiles() {
        return preserveExistingFiles;
    }

    /**
     * Property is not used anymore in the CopyStep and will be ignored.
     *
     * @deprecated Setting property 'preserveExistingFiles' on ArtifactCopyStep is deprecated and will be ignored.
     */
    @Deprecated
    public void setPreserveExistingFiles(final boolean preserveExistingFiles) {
        Deprecations.deprecated("**Deprecated** Setting property 'preserveExistingFiles' on ArtifactCopyStep is deprecated and will be ignored.");
        this.preserveExistingFiles = preserveExistingFiles;
    }


}
