package com.xebialabs.deployit.plugin.generic.step;

import java.util.*;
import java.util.stream.Collectors;


import com.google.common.base.Strings;
import com.xebialabs.deployit.io.SourceArtifactFile;
import com.xebialabs.deployit.io.copy.CopyUtil;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployableFolderArtifact;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.overthere.local.LocalFile;
import com.xebialabs.overthere.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.google.common.base.Strings.nullToEmpty;
import static com.google.common.collect.Sets.newHashSet;


@SuppressWarnings("serial")
public class ArtifactDeleteStep extends BaseFolderDeploymentStepSupport {

    private static final Logger logger =  LoggerFactory.getLogger(ArtifactDeleteStep.class);


    private String targetDirectory;

    private DeployableArtifact artifact;

    private String targetFile;

    private boolean targetDirectoryShared;

    private boolean targetPathSharedSubDirectories;

    private Set<String> deleteAdditionalTargetFiles = newHashSet();

    public ArtifactDeleteStep() {
    }

    public ArtifactDeleteStep(int order, HostContainer container, DeployableArtifact artifact, String targetDirectory) {
        super(order, container);
        this.artifact = artifact;
        this.targetDirectory = Objects.requireNonNull(targetDirectory);
    }

    public ArtifactDeleteStep(int order, HostContainer container, String targetDirectory) {
        this(order, container, null, targetDirectory);
    }

    @Override
    protected StepExitCode doExecute() throws Exception {
        if (logger.isDebugEnabled()) {
            logger.debug("[ArtifactDeleteStep] Starting doExecute. artifact: {}, targetDirectory: {}, targetFile: {}",
                artifact != null ? artifact.getName() : null, targetDirectory, targetFile);
        }
        if (artifact != null) {
            deleteArtifact();
        } else {
            deleteTargetFile();
        }
        deleteAdditionalFiles(getDeleteAdditionalTargetFiles());
        return StepExitCode.SUCCESS;
    }

    protected void deleteArtifact() {
        if (logger.isDebugEnabled()) {
            logger.debug("[ArtifactDeleteStep] deleteArtifact called. artifact: {}, targetDirectory: {}",
                artifact != null ? artifact.getName() : null, targetDirectory);
        }
        OverthereFile localFile = artifact.getFile();
        OverthereFile remoteDir = getRemoteConnection().getFile(targetDirectory);
        if (localFile.isDirectory()) {
            if (targetDirectoryShared) {
                deleteFilesAndFoldersForTargetShared(remoteDir, localFile);
            } else {
                logger.debug("[ArtifactDeleteStep] targetDirectoryShared is false, deleting remoteDir. remoteDir: {}", remoteDir.getPath());
                deleteFile(remoteDir);
            }
        } else {
            String name = nullToEmpty(targetFile).trim().isEmpty() ? artifact.getName() : targetFile;
            logger.debug("[ArtifactDeleteStep] Artifact is not a directory, deleting file: {}", name);
            deleteFile(remoteDir.getFile(name));
        }
    }

    protected void deleteFilesAndFoldersForTargetShared(OverthereFile remoteDir, OverthereFile localFile) {
        final BaseDeployableFolderArtifact baseDeployableFolderArtifact = getBaseDeployableFolderArtifact();
        if (isTypeIsFileFolder(baseDeployableFolderArtifact) && !isOneByOneCopyStrategy()) {
            logger.debug("[ArtifactDeleteStep] Artifact is a directory shared. isDeployableFolderArtifact: {}, isOneByOneCopyStrategy: {}", isDeployableFolderArtifact(), isOneByOneCopyStrategy());
            int stripComponents = artifact.hasProperty("stripComponents") ? Integer.parseInt(artifact.getProperty("stripComponents").toString()) : 0;
            Set<String> members = artifact.hasProperty("members") ? artifact.getProperty("members") : null;
            logger.debug("[ArtifactDeleteStep] targetDirectoryShared is true. stripComponents: {}, members: {}", stripComponents, members);
            if(stripComponents > 0 || (members != null && !members.isEmpty())) {
                logger.debug("[ArtifactDeleteStep] deleteFilesAndFolderOnTar called for isDeployableFolderArtifact: {}, isOneByOneCopyStrategy: {} isFileFolder {} with stripComponents {} / members size {}", isDeployableFolderArtifact(), isOneByOneCopyStrategy(), isTypeIsFileFolder(baseDeployableFolderArtifact), stripComponents , members);
                if (isFolderCanBeDeletedRecursive() && !targetPathSharedSubDirectories) {
                    deleteOnlyRootFilesAndFoldersRecursiveOnTar(stripComponents, members, remoteDir, localFile);
                } else {
                    logger.debug("[ArtifactDeleteStep] deleteFilesAndFolderOnTar recursive delete slower for isDeployableFolderArtifact: {}, isOneByOneCopyStrategy: {} isFileFolder {} targetPathSharedSubDirectories {}", isDeployableFolderArtifact(), isOneByOneCopyStrategy(), isTypeIsFileFolder(baseDeployableFolderArtifact), targetPathSharedSubDirectories);
                    deleteFilesAndFolderOnTar(stripComponents, members, remoteDir, localFile);
                }
            } else {
                logger.debug("[ArtifactDeleteStep] deleteRemoteDir called for isDeployableFolderArtifact: {}, isOneByOneCopyStrategy: {} isFileFolder {} WITH OUT with stripComponents {} / members size {}", isDeployableFolderArtifact(), isOneByOneCopyStrategy(), isTypeIsFileFolder(baseDeployableFolderArtifact),stripComponents, members != null ? members.size() : 0);
                deleteRemoteDirectory(localFile, remoteDir);
            }

        } else {
            logger.debug("[ArtifactDeleteStep] deleteRemoteDir for isDeployableFolderArtifact: {}, isOneByOneCopyStrategy: {} isFileFolder {}", isDeployableFolderArtifact(), isOneByOneCopyStrategy(), isTypeIsFileFolder(baseDeployableFolderArtifact));
            deleteRemoteDirectory(localFile, remoteDir);
        }
    }

    protected void deleteFilesAndFolderOnTar(int stripComponents, Set<String> members, OverthereFile remoteDir, OverthereFile localFile) {
        int stripComponentsDerived = stripComponents;
        Set<String> membersDerived = members;

        if (!isForceArchivedForFolder()) {
            stripComponentsDerived = stripComponents + 1;
            membersDerived = getMembersPathWithRootFolder(members, localFile.getName());
        }

        if (stripComponentsDerived > 0 || (membersDerived != null && !membersDerived.isEmpty())) {
            Set<String> archivePaths = extractPathsUsingTarCommand(stripComponentsDerived, membersDerived);
            deleteFilesAndFolders(archivePaths, remoteDir);
        } else {
            deleteRemoteDirectory(localFile, remoteDir);
        }
    }

    protected void deleteOnlyRootFilesAndFoldersRecursiveOnTar(int stripComponents, Set<String> members, OverthereFile remoteDir, OverthereFile localFile) {
        int stripComponentsDerived = stripComponents;
        Set<String> membersDerived = members;
        if (!isForceArchivedForFolder()) {
            stripComponentsDerived = stripComponents + 1 ;
            membersDerived = getMembersPathWithRootFolder(members, localFile.getName());
        }
        if (stripComponentsDerived > 0 || (membersDerived != null && !membersDerived.isEmpty())) {
            Set<String> archivePaths = extractPathsUsingTarCommand(stripComponentsDerived, membersDerived);
            deleteOnlyRootFilesAndFoldersRecursive(archivePaths, remoteDir);
        } else {
            deleteRemoteDirectory(localFile, remoteDir);
        }
    }

    protected boolean isTypeIsFileFolder(BaseDeployableFolderArtifact baseDeployableFolderArtifact) {
        return CopyUtil.isDeployableFileFolder(baseDeployableFolderArtifact);
    }

    protected Set<String> getMembersPathWithRootFolder(Set<String> members, String srcFileName) {
        if (members == null || members.isEmpty()) {
            return members;
        }
        return members.stream()
                .filter(member -> !member.trim().isEmpty())
                .map(member -> String.join("/", srcFileName, member))
                .collect(Collectors.toSet());
    }

    protected void deleteOnlyRootFilesAndFoldersRecursive(Set<String> archivePaths, OverthereFile remoteDir) {
        logger.debug("[ArtifactDeleteStep] deleteOnlyRootFilesAndFoldersRecursive called. archivePaths: {}, remoteDir: {}", archivePaths, remoteDir.getPath());
        Set<String> archivePathsWithoutRoot = archivePaths.stream()
                .filter( path -> path != null && !path.trim().isEmpty())
                .collect(Collectors.toCollection(LinkedHashSet::new));
        Set<String> processedPaths = new LinkedHashSet<>();

        Set<String> archivePathsOnlyRoots = getRootFoldersAndFiles(archivePathsWithoutRoot);

        if(!archivePathsOnlyRoots.isEmpty()) {
            archivePathsOnlyRoots.forEach(path -> {
                OverthereFile remoteDirToDel = remoteDir.getFile(path);
                boolean isFolderCanBeDeleted = remoteDirToDel.exists() && remoteDirToDel.isDirectory();
                if (isFolderCanBeDeleted) {
                    deleteFile(remoteDirToDel);
                    processedPaths.add(remoteDirToDel.getPath());
                    getCtx().logOutput("Deleted directory: " + remoteDirToDel.getPath());
                }
            });

            Set<String> filesAndDirectoriesPathNotProcessedOnlyRoot =
                    archivePathsWithoutRoot.stream()
                            .filter(path -> !processedPaths.contains(path))
                            .filter(path -> path != null && !path.trim().isEmpty() && !path.contains("/"))
                            .collect(Collectors.toCollection(LinkedHashSet::new));

            //delete only root files is enough as directories are already deleted recursively
            filesAndDirectoriesPathNotProcessedOnlyRoot.forEach(path -> {
                OverthereFile remoteDirFileToDelete = remoteDir.getFile(path);
                boolean isFileCanBeDeleted = remoteDirFileToDelete != null && remoteDirFileToDelete.exists() && remoteDirFileToDelete.isFile();
                if (isFileCanBeDeleted) {
                    deleteFile(remoteDirFileToDelete);
                    processedPaths.add(remoteDirFileToDelete.getPath());
                }
            });

            processedPathShouldNotBeEmpty(remoteDir, processedPaths);
        } else {
            logger.debug("[ArtifactDeleteStep] No root files or folders found in archivePathsWithoutRoot: {}", archivePathsWithoutRoot);
            deleteFilesAndFolders(archivePathsWithoutRoot, remoteDir);
        }
    }

    protected final Set<String> getRootFoldersAndFiles(Set<String> archivePathsWithoutRoot) {
        return archivePathsWithoutRoot.stream()
            .map(path -> path.contains("/") ? path.substring(0, path.indexOf("/")) : path)
            .collect(Collectors.toCollection(LinkedHashSet::new));
    }

    protected void deleteFilesAndFolders(Set<String> archivePaths, OverthereFile remoteDir) {
        logger.debug("[ArtifactDeleteStep]  deleteFilesAndFolders called. archivePaths: {}, remoteDir: {}", archivePaths, remoteDir.getPath());
        Set<String> archivePathsWithoutRoot = archivePaths.stream()
                .filter( path -> path != null && !path.trim().isEmpty())
                .collect(Collectors.toCollection(LinkedHashSet::new));

        Set<String> processedPaths = new LinkedHashSet<>();

        archivePathsWithoutRoot.forEach(path -> {
            OverthereFile remoteDirFileToDelete = remoteDir.getFile(path);
            boolean isFileCanBeDeleted = remoteDirFileToDelete != null && remoteDirFileToDelete.exists() && remoteDirFileToDelete.isFile();
            if (isFileCanBeDeleted) {
                deleteFile(remoteDirFileToDelete);
                processedPaths.add(remoteDirFileToDelete.getPath());
            }
        });

        Set<String> processedRelativePaths = processedPaths.stream()
            .map(absPath -> {
                String base = remoteDir.getPath();
                if (absPath.startsWith(base)) {
                    String rel = absPath.substring(base.length());
                    if (rel.startsWith("/")) rel = rel.substring(1);
                    return rel;
                }
                return absPath;
            })
            .collect(Collectors.toSet());

        Set<String> filesAndDirectoriesPathNotProcessed =
                archivePathsWithoutRoot.stream()
                        .filter(path -> !processedRelativePaths.contains(path))
                        .filter(path -> path != null && !path.trim().isEmpty())
                        .collect(Collectors.toCollection(LinkedHashSet::new));

        Set<String> rootDirectoriesPathNotProcessed =
                getRootFoldersAndFiles(filesAndDirectoriesPathNotProcessed);

        Set<String> allowedDirsAbs = null;

        if (targetPathSharedSubDirectories && !rootDirectoriesPathNotProcessed.isEmpty()) {
            allowedDirsAbs = getAllowedDirsToDelete(remoteDir, archivePaths);
            logger.debug("[ArtifactDeleteStep] deleteFolderIfEmpty called for allPaths {}, allowedDirsAbs: {}",  archivePaths, allowedDirsAbs);
        }

        for (String path : rootDirectoriesPathNotProcessed) {
            OverthereFile remoteDirToDel = remoteDir.getFile(path);
            boolean isFolderCanBeDeleted = remoteDirToDel.exists() && remoteDirToDel.isDirectory();
            if (isFolderCanBeDeleted) {
                if (targetPathSharedSubDirectories) {
                    deleteFolderIfEmpty(remoteDirToDel, processedPaths, allowedDirsAbs);
                } else {
                    deleteFile(remoteDirToDel);
                    processedPaths.add(remoteDirToDel.getPath());
                    getCtx().logOutput("Deleted directory: " + remoteDirToDel.getPath());
                }
            }
        }
        processedPathShouldNotBeEmpty(remoteDir, processedPaths);
    }

    protected Set<String> getAllowedDirsToDelete(OverthereFile remoteDir, Set<String> archivePaths) {
        Set<String> allowedDirsAbs = new LinkedHashSet<>();
        for (String path : archivePaths) {
            if (path != null && !path.trim().isEmpty()) {
                OverthereFile remoteDirToDelete = remoteDir.getFile(path);
                boolean isFolderExist = remoteDirToDelete != null && remoteDirToDelete.exists() && remoteDirToDelete.isDirectory();
                if (isFolderExist) {
                    allowedDirsAbs.add(remoteDirToDelete.getPath());
                }
            } else {
                logger.debug("[ArtifactDeleteStep] Skipping empty path in archivePaths: {}", path);
            }
        }
        return allowedDirsAbs;
    }

    private void processedPathShouldNotBeEmpty(OverthereFile remoteDir, Set<String> processedPaths) {
        if(processedPaths.isEmpty()) {
            getCtx().logError("No files or directories were deleted from " + remoteDir.getPath() + " on host " + getContainer().getHost());
            throw new RuntimeIOException("No files or directories were deleted from " + remoteDir.getPath() + " on host " + getContainer().getHost());
        } else {
            getCtx().logOutput("Deleted files and directories: " + String.join(", ", processedPaths) + " on host " + getContainer().getHost());
        }
    }

    private void deleteRemoteDirectory(final OverthereFile localFile, final OverthereFile remoteDir) {
        for (OverthereFile file : localFile.listFiles()) {
            OverthereFile remoteFile = remoteDir.getFile(file.getName());
            if (targetPathSharedSubDirectories && file.isDirectory()) {
                if (remoteFile.exists()) {
                    deleteRemoteDirectory(file, remoteFile);
                    if (remoteFile.listFiles().isEmpty()) {
                        deleteFile(remoteFile);
                    }
                } else {
                    getCtx().logOutput(remoteFile.getPath() + " does not exist on host " + getContainer().getHost() + ". Will not perform delete.");
                }
            } else {
                deleteFile(remoteFile);
            }
        }
    }

    protected void deleteTargetFile() {
        OverthereFile remoteFile = getRemoteConnection().getFile(targetDirectory);
        if (!nullToEmpty(targetFile).trim().isEmpty()) {
            remoteFile = remoteFile.getFile(targetFile);
        }
        deleteFile(remoteFile);
    }


    protected void deleteAdditionalFiles(Set<String> files) {
        for (String file : files) {
            OverthereFile remoteFile = getRemoteConnection().getFile(file);
            if (remoteFile.exists()) {
                getCtx().logOutput("Deleting " + remoteFile.getPath() + " on host " + getContainer().getHost());
                remoteFile.deleteRecursively();
            } else {
                getCtx().logOutput(remoteFile.getPath() + " does not exist on host " + getContainer().getHost() + ". Will not perform delete.");
            }
        }
    }

   protected void deleteFile(OverthereFile file) {
        getCtx().logOutput("Deleting " + file.getPath() + " on host " + getContainer().getHost());
        if (file.exists()) {
            file.deleteRecursively();
        } else {
            getCtx().logOutput(file.getPath() + " does not exist on host " + getContainer().getHost() + ". Will not perform delete.");
        }
    }

    protected void deleteFolderIfEmpty(OverthereFile folder, Set<String> processedPaths, Set<String> allowedDirsAbs) {
        getCtx().logOutput("Deleting " + folder.getPath() + " on host " + getContainer().getHost());
        if (!folder.exists()) {
            getCtx().logOutput(folder.getPath() + " does not exist on host " + getContainer().getHost() + ". Will not perform delete.");
            return;
        }
        // Always recurse into subdirectories to allow deeper allowed dirs to be processed
        List<OverthereFile> children = folder.listFiles();
        for (OverthereFile child : children) {
            if (child.exists() && child.isDirectory()) {
                deleteFolderIfEmpty(child, processedPaths, allowedDirsAbs);
            }
        }
        // Re-evaluate after recursion and delete only if allowed and empty
        List<OverthereFile> remaining = folder.listFiles();
        if (remaining.isEmpty()) {
            if (allowedDirsAbs != null && allowedDirsAbs.contains(folder.getPath())) {
                folder.deleteRecursively();
                getCtx().logOutput("Deleted directory: " + folder.getPath() + " on host " + getContainer().getHost());
                processedPaths.add(folder.getPath());
            } else {
                logger.debug("Skip deleting {} as it is not present in archive paths", folder.getPath());
            }
        } else {
            logger.debug("Skip deleting {} because it is not empty after deletion of subfolders", folder.getPath());
        }
    }

    protected Set<String> extractPathsUsingTarCommand(Integer stripComponents, Set<String> members) {
        LocalFile sourceArtifactLocalFolder = (LocalFile) LocalFile.valueOf(((SourceArtifactFile) artifact.getFile()).getLocalFile());
        LocalFile sourceArtifactLocalFolderAsTar = folderAsTar(sourceArtifactLocalFolder);

        CmdLine cmdLine = new CmdLine();
        cmdLine.addArgument("tar");
        cmdLine.addArgument("-tf");
        cmdLine.addArgument(sourceArtifactLocalFolderAsTar.getPath());

        // Add strip-components argument if specified
        if (stripComponents != null && stripComponents > 0) {
            cmdLine.addArgument("--strip-components=" + stripComponents);
        }

        // Add members to the command if specified
        if (members != null && !members.isEmpty()) {
            for (String member : members) {
                if (member.trim().isEmpty()) continue;
                cmdLine.addArgument(member);
            }
        }

        Set<String> paths = executeTarCommandAndExtractPath(stripComponents, cmdLine, sourceArtifactLocalFolderAsTar);
        logger.debug("[ArtifactDeleteStep] extractPathsUsingTarCommand called. stripComponents: {}, members: {}, paths: {}", stripComponents, members, paths);
        return paths;
    }

    protected Set<String> executeTarCommandAndExtractPath(Integer stripComponents, CmdLine cmdLine, LocalFile sourceArtifactLocalFolderAsTar) {
        Set<String> paths = new LinkedHashSet<>(); // Maintain insertion order and uniqueness
        try {
            StringBuilder stdout = new StringBuilder();
            StringBuilder stderr = new StringBuilder();

            OverthereExecutionOutputHandler stdoutHandler = createOutputHandler(stdout);
            OverthereExecutionOutputHandler stderrHandler = createOutputHandler(stderr);

            int exitCode = artifact.getFile().getConnection().execute(stdoutHandler, stderrHandler, cmdLine);
            if (exitCode != 0) {
                throw new RuntimeException("Error executing tar command. Exit code: " + exitCode + ". Error: " + stderr);
            }
            paths = getStrippedPaths(stripComponents, stdout);
        } catch (Exception e) {
            getCtx().logError("Error executing while fetch using tar command: " + e.getMessage());
        }
        if(paths.isEmpty()) {
            getCtx().logOutput("No files found in the tar archive to delete:" + sourceArtifactLocalFolderAsTar.getPath() + " on host " + getContainer().getHost());
        }
        return paths;
    }

    protected Set<String> getStrippedPaths(Integer stripComponents, StringBuilder stdout) {
        Set<String> paths = new LinkedHashSet<>();
        Arrays.stream(stdout.toString().split("\n"))
            .forEach(path -> {
                addPathByStripComponents(stripComponents, path, paths);
            });
        return paths;
    }

    private void addPathByStripComponents(Integer stripComponents, String path, Set<String> paths) {
        String[] components = path.split("/");
        if (stripComponents != null && stripComponents >= 0) {
            addStrippedPath(stripComponents, components, paths);
        } else {
            paths.add(path);
        }
    }

    private void addStrippedPath(Integer stripComponents, String[] components, Set<String> paths) {
        if (stripComponents < components.length) {
            paths.add(String.join("/", Arrays.copyOfRange(components, stripComponents, components.length)));
        }
    }

    private OverthereExecutionOutputHandler createOutputHandler(StringBuilder output) {
        return new OverthereExecutionOutputHandler() {
            @Override
            public void handleLine(String line) {
                output.append(line).append("\n");
            }

            @Override
            public void handleChar(char c) {
                // No implementation needed
            }
        };
    }

    @Override
    public String getDescription() {
        String description = super.getDescription();
        if (description == null) {
            return generateDescription();
        }
        return description;
    }

    @Override
    BaseDeployableFolderArtifact getBaseDeployableFolderArtifact() {
        return artifact instanceof BaseDeployableFolderArtifact ? (BaseDeployableFolderArtifact) artifact : null;
    }

    protected String generateDescription() {
        if (artifact != null) {
            return "Delete " + artifact.getName() + " from " + getContainer().getHost();
        }

        if (!Strings.nullToEmpty(targetFile).trim().isEmpty()) {
            return "Delete file " + targetFile + " from directory " + targetDirectory + " on host " + getContainer().getHost();
        }

        return "Delete directory " + targetDirectory + " from  " + getContainer().getHost();
    }

    public String getTargetFile() {
        return targetFile;
    }

    public void setTargetFile(String targetFile) {
        this.targetFile = targetFile;
    }

    public boolean isTargetDirectoryShared() {
        return targetDirectoryShared;
    }

    public void setTargetDirectoryShared(boolean targetDirectoryShared) {
        this.targetDirectoryShared = targetDirectoryShared;
    }

    public Set<String> getDeleteAdditionalTargetFiles() {
        return deleteAdditionalTargetFiles;
    }

    public void setDeleteAdditionalTargetFiles(Set<String> deleteAdditionalTargetFiles) {
        this.deleteAdditionalTargetFiles = deleteAdditionalTargetFiles;
    }

    public void setTargetPathSharedSubDirectories(final boolean targetPathSharedSubDirectories) {
        this.targetPathSharedSubDirectories = targetPathSharedSubDirectories;
    }
}
