package ai.digital.deploy.gitops.service;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.engine.api.ServiceHolder;
import ai.digital.deploy.gitops.DefaultGitOpsService;
import ai.digital.deploy.gitops.GitOpsService;
import ai.digital.deploy.gitops.GitUserCredentials;
import com.xebialabs.deployit.engine.api.XLDAsCodeService;
import com.xebialabs.deployit.engine.api.dto.XLDAsCodeResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GitOpsServiceHolder {
    
    private static final Logger logger = LoggerFactory.getLogger(GitOpsServiceHolder.class);
    private static final GitOpsServiceAdapter adapter = new GitOpsServiceAdapter();
    
    public static GitOpsServiceAdapter getGitOpsService() {
        return adapter;
    }
    
    public static XLDAsCodeService getService() {
        try {
            return ServiceHolder.getXLDAsCodeService();
        } catch (Exception e) {
            logger.debug("XLDAsCodeService not available: {}", e.getMessage());
            return null;
        }
    }
    
    // Adapter class to work with ConfigurationItem
    public static class GitOpsServiceAdapter {
        
        private final ai.digital.deploy.gitops.GitOpsService gitOpsService = new DefaultGitOpsService();
        
        /**
         * Helper class to hold extracted Git configuration from ConfigurationItem.
         */
        private static class GitConfig {
            final String url;
            final String branch;
            final GitUserCredentials credentials;
            
            GitConfig(ConfigurationItem gitSource) {
                this(gitSource, null);
            }
            
            GitConfig(ConfigurationItem gitSource, String branch) {
                this.url = (String) gitSource.getProperty("url");
                this.branch = branch;
                String username = (String) gitSource.getProperty("username");
                String password = (String) gitSource.getProperty("password");
                this.credentials = new GitUserCredentials(username, password);
            }
        }
        
        public void checkConnection(ConfigurationItem gitSource) {
            GitConfig config = new GitConfig(gitSource);
            gitOpsService.testConnection(config.url, null, config.credentials);
        }
        
        public String cloneRepository(ConfigurationItem gitSource, String targetDirectory) {
            GitConfig config = new GitConfig(gitSource);
            return gitOpsService.cloneRepository(config.url, config.branch, config.credentials, targetDirectory);
        }
        
        public String pullRepository(String repositoryPath, GitUserCredentials credentials) {
            return gitOpsService.pullRepository(repositoryPath, credentials);
        }
        
        public boolean isGitRepository(String directoryPath) {
            return gitOpsService.isGitRepository(directoryPath);
        }
        
        public String determineRepositoryPath(String url, String targetDirectory) {
            return gitOpsService.determineRepositoryPath(url, targetDirectory);
        }
        
        public XLDAsCodeResult apply(String yamlContent) {
            XLDAsCodeService xldAsCodeService = getService();
            if (xldAsCodeService != null) {
                try {
                    return xldAsCodeService.apply(yamlContent);
                } catch (Exception e) {
                    logger.error("Exception during YAML processing: {}", e.getMessage());
                    return new XLDAsCodeResult(
                        false, 
                        "Exception during YAML processing: " + e.getMessage(), 
                        null, 
                        e.getClass().getSimpleName() + ": " + e.getMessage()
                    );
                }
            } else {
                return new XLDAsCodeResult(
                    false, 
                    "XLDAsCodeService not available in this environment", 
                    null, 
                    "ServiceHolder.getXLDAsCodeService() returned null"
                );
            }
        }
        
        /**
         * Generates YAML definitions from existing XL Deploy repository structure.
         * This method provides external worker support by delegating to XLDAsCodeService.
         *
         * @param path Repository path to generate from
         * @param globalPermissions Include global permissions in generation
         * @param roles Include roles in generation
         * @param users Include users in generation
         * @param includeSecrets Include secrets in generation
         * @param includeDefaults Include default values in generation
         * @return XLDAsCodeResult containing generated YAML content
         */
        public XLDAsCodeResult generate(String path, boolean globalPermissions, boolean roles,
                                        boolean users, boolean includeSecrets, boolean includeDefaults) {
            XLDAsCodeService xldAsCodeService = getService();
            if (xldAsCodeService != null) {
                try {
                    return xldAsCodeService.generate(path, globalPermissions, roles, users, includeSecrets, includeDefaults);
                } catch (Exception e) {
                    logger.error("Exception during YAML generation: {}", e.getMessage());
                    return new XLDAsCodeResult(
                        false, 
                        "Exception during YAML generation: " + e.getMessage(), 
                        null, 
                        e.getClass().getSimpleName() + ": " + e.getMessage()
                    );
                }
            } else {
                return new XLDAsCodeResult(
                    false, 
                    "XLDAsCodeService not available in this environment", 
                    null, 
                    "ServiceHolder.getXLDAsCodeService() returned null"
                );
            }
        }

    }
}