package ai.digital.deploy.gitops.step;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.Status;
import org.eclipse.jgit.api.AddCommand;
import org.eclipse.jgit.api.CommitCommand;
import org.eclipse.jgit.api.PushCommand;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;
import org.eclipse.jgit.transport.PushResult;
import org.eclipse.jgit.transport.RemoteRefUpdate;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Set;

/**
 * Step to commit generated YAML files and push them to the Git repository.
 * This step stages infrastructure.yaml and environment.yaml files,
 * commits them with a descriptive message, and pushes to the remote repository.
 */
public class GitCommitAndPushStep implements Step {

    private static final String INFRASTRUCTURE_YAML = "infrastructure.yaml";
    private static final String ENVIRONMENT_YAML = "environment.yaml";
    
    private final String gitDirectoryId;
    private final String username;
    private final String password;
    private final String tagName;

    public GitCommitAndPushStep(String gitDirectoryId, String username, String password, String tagName) {
        this.gitDirectoryId = gitDirectoryId;
        this.username = username;
        this.password = password;
        this.tagName = tagName;
    }

    @Override
    public String getDescription() {
        return "Commit and Push to Git";
    }

    @Override
    public int getOrder() {
        return 40;
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        ctx.logOutput("Starting Git commit and push");
        
        Object localRepoDirAttr = ctx.getAttribute(GitOpsExecutionAttributes.LOCAL_REPO_DIR);
        if (localRepoDirAttr == null) {
            ctx.logError("Local repository directory not found in context. Did the repository validation step run?");
            return StepExitCode.FAIL;
        }
        
        Object targetDirAttr = ctx.getAttribute(GitOpsExecutionAttributes.TARGET_DIRECTORY);
        if (targetDirAttr == null) {
            ctx.logError("Target directory not found in context. Did the repository validation step run?");
            return StepExitCode.FAIL;
        }

        File localRepoDir = new File(localRepoDirAttr.toString());
        Path targetDirectory = Paths.get(targetDirAttr.toString());
        
        ctx.logOutput("Local repository: " + localRepoDir.getAbsolutePath());
        ctx.logOutput("Target directory: " + targetDirectory);
        
        Git git = null;
        try {
            git = Git.open(localRepoDir);
            
            // Calculate relative paths from repo root for staging
            Path repoPath = localRepoDir.toPath();
            Path relativeTarget = repoPath.relativize(targetDirectory);
            
            String infrastructureRelPath = relativeTarget.resolve(INFRASTRUCTURE_YAML).toString().replace('\\', '/');
            String environmentRelPath = relativeTarget.resolve(ENVIRONMENT_YAML).toString().replace('\\', '/');
            
            ctx.logOutput("Staging files...");
            ctx.logOutput("  - " + infrastructureRelPath);
            ctx.logOutput("  - " + environmentRelPath);
            
            // Stage the generated files
            AddCommand addCommand = git.add();
            addCommand.addFilepattern(infrastructureRelPath);
            addCommand.addFilepattern(environmentRelPath);
            addCommand.call();
            
            // Check if there are changes to commit
            Status status = git.status().call();
            // Create a mutable set from the unmodifiable sets returned by status
            Set<String> staged = new java.util.HashSet<>(status.getAdded());
            staged.addAll(status.getChanged());
            
            if (staged.isEmpty() && status.getModified().isEmpty()) {
                ctx.logOutput("No changes detected. Nothing to commit.");
                return StepExitCode.SUCCESS;
            }
            
            ctx.logOutput("Changes staged for commit:");
            for (String file : staged) {
                ctx.logOutput("  Added/Changed: " + file);
            }
            for (String file : status.getModified()) {
                // Stage modified files as well
                git.add().addFilepattern(file).call();
                ctx.logOutput("  Modified: " + file);
            }
            
            // Create commit message with timestamp
            String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
            
            // Use the tag name passed from the delegate (already validated)
            ctx.logOutput("Using tag: " + tagName);
            
            String commitMessage = String.format("GitOps Export [%s]: Updated infrastructure and environment YAMLs\n\n" +
                "Tag: %s\n" +
                "Exported from XL Deploy at %s\n" +
                "Git Directory: %s", tagName, tagName, timestamp, gitDirectoryId);
            
            ctx.logOutput("Committing changes...");
            CommitCommand commitCommand = git.commit();
            commitCommand.setMessage(commitMessage);
            
            // Set author if username is available
            if (username != null && !username.isEmpty()) {
                commitCommand.setAuthor(username, username + "@xldeploy.local");
                commitCommand.setCommitter(username, username + "@xldeploy.local");
            }
            
            org.eclipse.jgit.revwalk.RevCommit commit = commitCommand.call();
            ctx.logOutput("Commit successful: " + commit.getId().getName());
            
            // Create a tag for this export (already validated in delegate)
            String tagMessage = String.format("XLD Export at %s", timestamp);
            
            ctx.logOutput("Creating tag: " + tagName);
            git.tag()
                .setName(tagName)
                .setMessage(tagMessage)
                .setAnnotated(true)
                .call();
            ctx.logOutput("Tag created successfully: " + tagName);
            
            // Push to remote (including tags)
            ctx.logOutput("Pushing to remote repository...");
            PushCommand pushCommand = git.push();
            
            if (username != null && !username.isEmpty() && password != null && !password.isEmpty()) {
                pushCommand.setCredentialsProvider(
                    new UsernamePasswordCredentialsProvider(username, password)
                );
            }
            
            Iterable<PushResult> pushResults = pushCommand.call();
            
            // Check push results
            boolean pushSuccess = true;
            for (PushResult pushResult : pushResults) {
                for (RemoteRefUpdate update : pushResult.getRemoteUpdates()) {
                    RemoteRefUpdate.Status updateStatus = update.getStatus();
                    ctx.logOutput("Push status for " + update.getRemoteName() + ": " + updateStatus);
                    
                    if (updateStatus != RemoteRefUpdate.Status.OK && 
                        updateStatus != RemoteRefUpdate.Status.UP_TO_DATE) {
                        pushSuccess = false;
                        ctx.logError("Push failed: " + update.getMessage());
                    }
                }
            }
            
            // Push tags separately
            ctx.logOutput("Pushing tags to remote repository...");
            PushCommand pushTagsCommand = git.push().setPushTags();
            
            if (username != null && !username.isEmpty() && password != null && !password.isEmpty()) {
                pushTagsCommand.setCredentialsProvider(
                    new UsernamePasswordCredentialsProvider(username, password)
                );
            }
            
            Iterable<PushResult> tagPushResults = pushTagsCommand.call();
            
            for (PushResult pushResult : tagPushResults) {
                for (RemoteRefUpdate update : pushResult.getRemoteUpdates()) {
                    RemoteRefUpdate.Status updateStatus = update.getStatus();
                    ctx.logOutput("Tag push status for " + update.getRemoteName() + ": " + updateStatus);
                    
                    if (updateStatus != RemoteRefUpdate.Status.OK && 
                        updateStatus != RemoteRefUpdate.Status.UP_TO_DATE) {
                        ctx.logOutput("Warning: Tag push may have issues: " + update.getMessage());
                    }
                }
            }
            
            if (pushSuccess) {
                ctx.logOutput("Successfully pushed changes and tag '" + tagName + "' to remote repository");
                return StepExitCode.SUCCESS;
            } else {
                ctx.logError("Push to remote repository failed");
                return StepExitCode.FAIL;
            }
            
        } catch (Exception e) {
            ctx.logError("Git commit and push failed: " + e.getMessage(), e);
            return StepExitCode.FAIL;
        } finally {
            if (git != null) {
                git.close();
            }
        }
    }
}
