package ai.digital.deploy.gitops.step;

import ai.digital.deploy.gitops.service.GitOpsServiceHolder;
import ai.digital.deploy.gitops.util.AsCodeResponseLogger;
import ai.digital.deploy.gitops.util.SCMInfo;
import ai.digital.deploy.gitops.util.SCMInfoExtractor;
import ai.digital.deploy.gitops.util.YamlKindFilter;
import ai.digital.deploy.gitops.util.YamlKindFilter.FilterResult;
import com.xebialabs.deployit.engine.api.ServiceHolder;
import com.xebialabs.deployit.engine.api.XLDAsCodeService;
import com.xebialabs.deployit.engine.api.dto.XLDAsCodeResult;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

/**
 * Step to import Infrastructure CIs from infrastructure.yaml file.
 * This step looks specifically for infrastructure.yaml in the target directory.
 * If the file is not found, the step fails and the user must manually skip it.
 */
public class GitApplyInfrastructureStep implements Step {

    private static final String INFRASTRUCTURE_YAML = "infrastructure.yaml";
    
    private final String gitDirectoryId;

    public GitApplyInfrastructureStep(String gitDirectoryId) {
        this.gitDirectoryId = gitDirectoryId;
    }

    @Override
    public String getDescription() {
        return "Import Infrastructure";
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        ctx.logOutput("Starting Infrastructure import validation and apply");
        
        Object directoryAttr = ctx.getAttribute(GitOpsExecutionAttributes.TARGET_DIRECTORY);
        if (directoryAttr == null) {
            ctx.logError("Target directory not found in context. Did the repository validation step run?");
            return StepExitCode.FAIL;
        }

        Path directoryPath = Paths.get(directoryAttr.toString());
        
        // Validate directory exists
        if (!Files.exists(directoryPath) || !Files.isDirectory(directoryPath)) {
            ctx.logError("Directory missing or invalid: " + directoryPath);
            return StepExitCode.FAIL;
        }
        
        Path infrastructureYamlPath = directoryPath.resolve(INFRASTRUCTURE_YAML);
        
        ctx.logOutput("Looking for infrastructure.yaml in: " + directoryPath);

        if (!Files.exists(infrastructureYamlPath)) {
            ctx.logError("infrastructure.yaml not found in directory: " + directoryPath);
            ctx.logError("Please ensure the file exists or skip this step manually.");
            return StepExitCode.FAIL;
        }

        ctx.logOutput("Found infrastructure.yaml: " + infrastructureYamlPath);
        
        return applyYamlFile(infrastructureYamlPath, ctx);
    }

    private StepExitCode applyYamlFile(Path yamlFile, ExecutionContext ctx) {
        XLDAsCodeService xldAsCodeService = ServiceHolder.getXLDAsCodeService();
        GitOpsServiceHolder.GitOpsServiceAdapter adapter = null;

        if (xldAsCodeService == null) {
            ctx.logOutput("XLDAsCodeService not available. Falling back to adapter.");
            adapter = GitOpsServiceHolder.getGitOpsService();
        }

        try {
            String yamlContent = new String(Files.readAllBytes(yamlFile));

            if (yamlContent.trim().isEmpty()) {
                ctx.logOutput("infrastructure.yaml is empty. Skipping.");
                return StepExitCode.SUCCESS;
            }

            // Check for multiple spec.directory entries and warn
            checkMultipleSpecDirectories(yamlContent, ctx);
            
            // Filter YAML content to only include Infrastructure kind
            ctx.logOutput("Validating YAML content for Infrastructure kind...");
            FilterResult filterResult = YamlKindFilter.filterByKind(
                yamlContent, YamlKindFilter.KIND_INFRASTRUCTURE, ctx);
            
            // Log any skipped documents
            YamlKindFilter.logSkippedDocuments(filterResult, INFRASTRUCTURE_YAML, 
                YamlKindFilter.KIND_INFRASTRUCTURE, ctx);
            
            ctx.logOutput("  Found " + filterResult.getMatchingDocuments() + " of " + 
                         filterResult.getTotalDocuments() + " document(s) with kind: Infrastructure");
            
            if (!filterResult.hasContent()) {
                ctx.logError("No Infrastructure documents found in infrastructure.yaml.");
                ctx.logError("The file must contain at least one document with 'kind: Infrastructure'.");
                return StepExitCode.FAIL;
            }

            ctx.logOutput("Applying infrastructure.yaml (Infrastructure kind only)...");
            
            // Get SCM traceability data from context and create file-specific copy
            SCMInfo scmData = getSCMDataForFile(ctx, INFRASTRUCTURE_YAML);
            
            XLDAsCodeResult result;
            if (xldAsCodeService != null) {
                if (scmData != null && scmData.hasCommit()) {
                    ctx.logOutput("Applying with SCM traceability (commit: " + 
                                 scmData.getCommit().substring(0, Math.min(7, scmData.getCommit().length())) + ")");
                    result = xldAsCodeService.applyWithScm(
                        filterResult.getFilteredContent(),
                        scmData.getKind(),
                        scmData.getCommit(),
                        scmData.getAuthor(),
                        scmData.getDate(),
                        scmData.getMessage(),
                        scmData.getRemote(),
                        scmData.getFileName()
                    );
                } else {
                    result = xldAsCodeService.apply(filterResult.getFilteredContent());
                }
            } else {
                // Fallback adapter doesn't support SCM traceability
                result = adapter.apply(filterResult.getFilteredContent());
            }

            if (result.isSuccess()) {
                ctx.logOutput("Successfully applied infrastructure.yaml");
                // Log Created/Updated/Deleted changes like xl CLI
                AsCodeResponseLogger.logChanges(result, ctx);
                return StepExitCode.SUCCESS;
            } else {
                ctx.logError("Failed to apply infrastructure.yaml");
                ctx.logError("Reason: " + result.getMessage());
                if (result.getError() != null) {
                    ctx.logError("Error: " + result.getError());
                }
                return StepExitCode.FAIL;
            }

        } catch (Exception e) {
            ctx.logError("Exception processing infrastructure.yaml: " + e.getMessage());
            return StepExitCode.FAIL;
        }
    }

    /**
     * Gets SCM traceability data from context and creates a file-specific copy with the filename set.
     * 
     * @param ctx The execution context
     * @param fileName The name of the YAML file being applied
     * @return SCMInfo with filename set, or null if not available
     */
    private SCMInfo getSCMDataForFile(ExecutionContext ctx, String fileName) {
        Object scmAttr = ctx.getAttribute(GitOpsExecutionAttributes.SCM_TRACEABILITY_DATA);
        if (scmAttr instanceof SCMInfo) {
            SCMInfo baseSCM = (SCMInfo) scmAttr;
            // Create a new SCMInfo with the specific filename
            return baseSCM.withFileName(fileName);
        }
        
        // Try to create minimal SCM data with just the remote URL if available
        Object remoteUrlAttr = ctx.getAttribute(GitOpsExecutionAttributes.GIT_REMOTE_URL);
        if (remoteUrlAttr != null) {
            return SCMInfoExtractor.createMinimalSCMInfo(remoteUrlAttr.toString(), fileName);
        }
        
        return null;
    }

    /**
     * Checks for multiple spec.directory entries in the YAML content and logs a warning if found.
     * Counts occurrences of '- directory:' pattern which represents list items with directory field.
     */
    private void checkMultipleSpecDirectories(String yamlContent, ExecutionContext ctx) {
        // Count occurrences of list items with 'directory:' field (e.g., "- directory: some/path")
        java.util.regex.Pattern pattern = java.util.regex.Pattern.compile(
            "^\\s*-\\s*directory\\s*:", 
            java.util.regex.Pattern.MULTILINE);
        java.util.regex.Matcher matcher = pattern.matcher(yamlContent);
        
        int count = 0;
        while (matcher.find()) {
            count++;
        }
        
        if (count > 1) {
            ctx.logError("Warning: Found " + count + " spec.directory entries in infrastructure.yaml. " +
                         "Multiple directory specifications detected. Proceeding with import.");
        }
    }

    @Override
    public int getOrder() {
        return 20;
    }
}
