package ai.digital.deploy.gitops.step;

import ai.digital.deploy.gitops.GitOpsDelegate;
import ai.digital.deploy.gitops.util.YamlHandler;
import com.xebialabs.deployit.engine.api.XLDAsCodeService;
import com.xebialabs.deployit.engine.api.dto.XLDAsCodeResult;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.rules.StepMetadata;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import ai.digital.deploy.gitops.service.GitOpsServiceHolder;
import com.xebialabs.xlplatform.satellite.Satellite;
import com.xebialabs.xlplatform.satellite.SatelliteAware;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.List;

@StepMetadata(name = "xldAsCodeApply")
public class GitOpsApplyStep implements SatelliteAware {
    
    private static final Logger logger = LoggerFactory.getLogger(GitOpsApplyStep.class);
    
    private ConfigurationItem gitSource;
    private String repositoryPath;
    
    // Default constructor required by XL Deploy rule scanning
    public GitOpsApplyStep() {
    }
    
    public GitOpsApplyStep(ConfigurationItem gitSource, String repositoryPath) {
        this.gitSource = gitSource;
        this.repositoryPath = repositoryPath;
    }
    
    @Override
    public String getDescription() {
        return "Apply YAML files through XLDAsCodeService";
    }
    
    @Override
    public int getOrder() {
        return 60;
    }
    
    @Override
    public Satellite getSatellite() {
        // Run locally on the XL Deploy server
        return null;
    }
    
    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        ctx.logOutput("Starting apply operation");
        ctx.logOutput("Repository path: " + repositoryPath);
        
        try {
            if (!validateRepositoryPath(ctx)) {
                return StepExitCode.SUCCESS;
            }
            
            ctx.logOutput("Scanning and reading YAML files...");
            List<YamlHandler.YamlContent> yamlContents = YamlHandler.scanAndReadYamlFiles(repositoryPath);
            
            if (yamlContents.isEmpty()) {
                ctx.logOutput("No YAML files found in repository");
                return StepExitCode.SUCCESS;
            }
            
            List<YamlHandler.YamlContent> validContents = yamlContents.stream()
                .filter(YamlHandler.YamlContent::isSuccess)
                .toList();
            
            ctx.logOutput("Found " + validContents.size() + " valid YAML files to process");
            
            if (validContents.isEmpty()) {
                ctx.logOutput("No valid YAML content to process");
                return StepExitCode.SUCCESS;
            }
            
            // Sort YAML files by processing order: Infrastructure → Environment → Applications
            List<YamlHandler.YamlContent> sortedContents = sortYamlFilesByProcessingOrder(validContents);
            ctx.logOutput("Processing " + sortedContents.size() + " YAML files...");
            int successCount = 0;
            int failureCount = 0;
            
            XLDAsCodeService xldAsCodeService = getXLDAsCodeService();
            GitOpsServiceHolder.GitOpsServiceAdapter adapter = null;
            
            if (xldAsCodeService == null) {
                // External worker mode - use GitOpsServiceHolder adapter
                ctx.logOutput("External worker mode - using adapter approach");
                adapter = GitOpsServiceHolder.getGitOpsService();
            }
            
            for (YamlHandler.YamlContent yamlContent : sortedContents) {
                String relativePath = YamlHandler.getRelativePathForDisplay(repositoryPath, yamlContent.getFilePath());
                
                try {
                    if (yamlContent.getContent().trim().isEmpty()) {
                        ctx.logOutput("  Skipped (empty file): " + relativePath);
                        continue;
                    }
                    
                    XLDAsCodeResult result = (xldAsCodeService != null) 
                        ? xldAsCodeService.apply(yamlContent.getContent())
                        : adapter.apply(yamlContent.getContent());
                    
                    if (result.isSuccess()) {
                        ctx.logOutput("Success: " + relativePath);
                        if (result.getMessage() != null) {
                            ctx.logOutput("    " + result.getMessage());
                        }
                        successCount++;
                    } else {
                        ctx.logOutput("Failed: " + relativePath);
                        ctx.logOutput("Reason: " + result.getMessage());
                        
                        if (result.getError() != null) {
                            ctx.logError("Error Details: " + result.getError());
                        }
                        failureCount++;
                    }
                    
                } catch (Exception e) {
                    ctx.logOutput("Exception: " + relativePath + " - " + e.getMessage());
                    logger.error("Exception processing YAML file: {}", yamlContent.getFilePath(), e);
                    failureCount++;
                }
            }
            
            ctx.logOutput("Processing complete: " + successCount + " successful, " + failureCount + " failed");
            
            return successCount > 0 ? StepExitCode.SUCCESS : StepExitCode.FAIL;
            
        } catch (Exception e) {
            ctx.logError("Error during apply operation: " + e.getMessage());
            logger.error("Error in GitOpsApplyStep", e);
            return StepExitCode.FAIL;
        }
    }
    
    private boolean validateRepositoryPath(ExecutionContext ctx) {
        if (repositoryPath == null || repositoryPath.trim().isEmpty()) {
            ctx.logOutput("Repository path is not specified - will attempt to find cloned repository");
            repositoryPath = findActualRepositoryPath(ctx);
            if (repositoryPath == null) {
                ctx.logOutput("Could not locate cloned repository");
                return false;
            }
        }
        
        File repoDir = new File(repositoryPath);
        if (!repoDir.exists()) {
            ctx.logOutput("Repository path does not exist: " + repositoryPath);
            // Try to find the actual cloned repository
            String actualPath = findActualRepositoryPath(ctx);
            if (actualPath != null) {
                ctx.logOutput("Found actual repository at: " + actualPath);
                repositoryPath = actualPath;
                repoDir = new File(repositoryPath);
            } else {
                return false;
            }
        }
        
        if (!repoDir.isDirectory()) {
            ctx.logOutput("Repository path is not a directory: " + repositoryPath);
            return false;
        }
        
        ctx.logOutput("Repository path validated: " + repositoryPath);
        return true;
    }
    
    /**
     * Determines repository path by searching in work/git_repos directory.
     */
    private String findActualRepositoryPath(ExecutionContext ctx) {
        if (gitSource == null) {
            return null;
        }
        
        try {
            // Search in work/git_repos directory under server home
            return searchInGitReposDirectory(ctx);
            
        } catch (Exception e) {
            ctx.logOutput("Error while determining repository path: " + e.getMessage());
            return null;
        }
    }
    
    /**
     * Search in work/git_repos directory for the cloned repository.
     */
    private String searchInGitReposDirectory(ExecutionContext ctx) {
        try {
            String serverHome = System.getProperty("user.dir");
            File gitReposDir = new File(serverHome, "work" + File.separator + "git_repos");
            
            if (!gitReposDir.exists() || !gitReposDir.isDirectory()) {
                ctx.logOutput("Git repos directory not found: " + gitReposDir.getAbsolutePath());
                return null;
            }
            
            ctx.logOutput("Searching for Git repositories in: " + gitReposDir.getAbsolutePath());
            
            // Look for directories in git_repos
            File[] repoDirs = gitReposDir.listFiles(File::isDirectory);
            
            if (repoDirs == null || repoDirs.length == 0) {
                ctx.logOutput("No directories found in git_repos");
                return null;
            }
            
            ctx.logOutput("Found " + repoDirs.length + " directories in git_repos");
            
            // Find the most recently modified directory
            File mostRecent = repoDirs[0];
            for (File dir : repoDirs) {
                if (dir.lastModified() > mostRecent.lastModified()) {
                    mostRecent = dir;
                }
            }
            
            ctx.logOutput("Most recent repository directory: " + mostRecent.getAbsolutePath());
            
            // Check if it's a valid Git repository
            if (isValidGitRepository(mostRecent.getAbsolutePath())) {
                ctx.logOutput("Found Git repository at: " + mostRecent.getAbsolutePath());
                return mostRecent.getAbsolutePath();
            }
            
            // Check subdirectories for the actual cloned repository
            File[] subdirs = mostRecent.listFiles(File::isDirectory);
            if (subdirs != null && subdirs.length > 0) {
                for (File subdir : subdirs) {
                    if (isValidGitRepository(subdir.getAbsolutePath())) {
                        ctx.logOutput("Found Git repository at: " + subdir.getAbsolutePath());
                        return subdir.getAbsolutePath();
                    }
                }
            }
            
            ctx.logOutput("No valid Git repository found in: " + mostRecent.getAbsolutePath());
            
        } catch (Exception e) {
            ctx.logOutput("Error searching temp directories: " + e.getMessage());
        }
        
        return null;
    }
    
    /**
     * Checks if a path points to a valid Git repository.
     */
    private boolean isValidGitRepository(String path) {
        if (path == null || path.trim().isEmpty()) {
            return false;
        }
        
        try {
            File dir = new File(path);
            if (!dir.exists() || !dir.isDirectory()) {
                return false;
            }
            
            File gitDir = new File(dir, ".git");
            return gitDir.exists() && gitDir.isDirectory();
            
        } catch (Exception e) {
            return false;
        }
    }
    
    private XLDAsCodeService getXLDAsCodeService() {
        try {
            XLDAsCodeService service = GitOpsServiceHolder.getService();
            
            if (service != null) {
                String className = service.getClass().getName();
                
                // Check if this is a REST proxy (external worker) 
                if (className.contains("$Proxy") || className.contains("jdk.proxy")) {
                    // Return null to trigger external worker processing
                    return null;
                } else {
                    // It's an in-process service
                    return service;
                }
            }
            
            return null;
        } catch (Exception e) {
            logger.debug("ServiceHolder not available: {}", e.getMessage());
            return null;
        }
    }
    

    
    /**
     * Sorts YAML files by processing order to handle dependencies:
     * 1. Infrastructure (servers, hosts, etc.)
     * 2. Environment (deployment targets that reference infrastructure)
     * 3. Applications (deployment packages that target environments)
     */
    private List<YamlHandler.YamlContent> sortYamlFilesByProcessingOrder(List<YamlHandler.YamlContent> yamlContents) {
        return yamlContents.stream()
            .sorted((a, b) -> {
                int orderA = getProcessingOrder(a);
                int orderB = getProcessingOrder(b);
                
                if (orderA != orderB) {
                    return Integer.compare(orderA, orderB);
                }
                
                // Within same category, sort alphabetically by file path
                return a.getFilePath().compareTo(b.getFilePath());
            })
            .toList();
    }
    
    /**
     * Determines processing order: Infrastructure(1) → Dictionaries(2) → Environment(3) → Applications(4)
     */
    private int getProcessingOrder(YamlHandler.YamlContent yamlContent) {
        String filePath = yamlContent.getFilePath().toLowerCase().replace('\\', '/');
        String content = yamlContent.getContent().toLowerCase();
        
        // Dictionaries have priority regardless of location
        if (content.contains("type: udm.dictionary")) {
            return 1;
        }
        
        // Check by directory or kind
        if (filePath.contains("/infrastructure/") || content.contains("kind: infrastructure")) {
            return 2;
        }
        
        if (filePath.contains("/environments/") || content.contains("kind: environments")) {
            return 3;
        }
        
        if (filePath.contains("/applications/") || content.contains("kind: applications")) {
            return 4;
        }
        
        return 5; // Default for other files
    }
    

    

}