package ai.digital.deploy.gitops.step;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

public class GitYamlValidationStep implements Step {

    private final String gitDirectoryId;

    public GitYamlValidationStep(String gitDirectoryId) {
        this.gitDirectoryId = gitDirectoryId;
    }

    @Override
    public String getDescription() {
        return "Validating YAML";
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        Object directoryAttr = ctx.getAttribute(GitOpsExecutionAttributes.TARGET_DIRECTORY);
        if (directoryAttr == null) {
            ctx.logError("Target directory not found in context. Did the repository validation step run?");
            return StepExitCode.FAIL;
        }

        Path directoryPath = Paths.get(directoryAttr.toString());
        ctx.logOutput("Validating YAML files under: " + directoryPath);

        if (!Files.exists(directoryPath) || !Files.isDirectory(directoryPath)) {
            ctx.logError("Directory missing or invalid: " + directoryPath);
            return StepExitCode.FAIL;
        }

        List<String> yamlFiles = new ArrayList<>();
        try (Stream<Path> files = Files.walk(directoryPath)) {
            files.filter(Files::isRegularFile)
                 .filter(p -> p.toString().endsWith(".yaml") || p.toString().endsWith(".yml"))
                 .forEach(p -> {
                     yamlFiles.add(p.toString());
                     ctx.logOutput("Found YAML file: " + p.getFileName());
                 });
        }

        if (yamlFiles.isEmpty()) {
            ctx.logOutput("No YAML files found under directory: " + directoryPath);
            ctx.logOutput("Individual import steps will check for specific files.");
        } else {
            ctx.logOutput("Total YAML files found: " + yamlFiles.size());
        }

        ctx.setAttribute(GitOpsExecutionAttributes.YAML_FILES, yamlFiles);
        return StepExitCode.SUCCESS;
    }

    @Override
    public int getOrder() {
        return 20;
    }
}
