package ai.digital.deploy.gitops;

import java.util.List;
import java.util.Map;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Delegate;
import com.xebialabs.deployit.plugin.api.udm.Parameters;
import ai.digital.deploy.gitops.step.CheckGitConnectionStep;
import ai.digital.deploy.gitops.step.GitCloneStep;
import ai.digital.deploy.gitops.step.GitOpsApplyStep;
import ai.digital.deploy.gitops.util.YamlHandler;

public final class GitOpsDelegate {

    private GitOpsDelegate() {}

    @Delegate(name = "gitOpsCheckConnection")
    public static List<Step> checkConnection(ConfigurationItem item, String name, Map<String, String> args, Parameters params) {
        return Lists.<Step>newArrayList(new CheckGitConnectionStep(item));
    }

    @Delegate(name = "gitOpsImport")
    public static List<Step> importRepository(ConfigurationItem item, String name, Map<String, String> args, Parameters params) {
        // Get target directory from configuration item's targetPath property
        String targetDirectory = (String) item.getProperty("targetPath");
        
        // If targetPath is empty or null, use temp directory (Stitch pattern)
        if (targetDirectory == null || targetDirectory.trim().isEmpty()) {
            targetDirectory = null; // This will trigger temp directory creation in GitCloneStep
        }
        
        // Clone repository and then scan for YAML files to create CIs
        List<Step> steps = Lists.<Step>newArrayList(new GitCloneStep(item, targetDirectory));
        
        // Add YAML processing step after cloning
        try {
            String repositoryPath = determineRepositoryPath(item, targetDirectory);
            steps.add(new GitOpsApplyStep(item, repositoryPath));
        } catch (Exception e) {
            // If we can't determine the path, just log and continue with clone only
            System.out.println("Warning: Could not determine repository path for CI scanning: " + e.getMessage());
        }
        
        return steps;
    }
    
    /**
     * Determines the repository path where the repository will be or was cloned.
     * Handles both specific target directories and temporary directories.
     * Uses the same YamlHandler utility to ensure consistency.
     */
    public static String determineRepositoryPath(ConfigurationItem item, String targetDirectory) {
        if (targetDirectory != null && !targetDirectory.trim().isEmpty()) {
            // Repository will be cloned to a specific target directory
            String url = (String) item.getProperty("url");
            if (url == null) {
                throw new RuntimeException("Git URL not specified");
            }
            
            // Use the same logic as DefaultGitOpsService for consistency
            String orgPrefixedRepoName = YamlHandler.createOrgPrefixedRepositoryName(url);
            
            return targetDirectory + java.io.File.separator + orgPrefixedRepoName;
        } else {
            // Repository will be cloned to a temporary directory
            // For temp directories, we pass null and let GitOpsApplyStep handle it
            // by working with the GitCloneStep to get the actual cloned path
            return null;
        }
    }
}