package ai.digital.deploy.gitops.step;

import ai.digital.deploy.gitops.util.SCMInfo;
import ai.digital.deploy.gitops.util.SCMInfoExtractor;
import ai.digital.deploy.gitops.util.YamlHandler;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.PullResult;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevTree;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;
import org.eclipse.jgit.treewalk.TreeWalk;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

public class GitRepositoryValidationStep implements Step {

    private final String url;
    private final String username;
    private final String password;
    private final String gitRepositoryPath;
    private final String gitDirectoryId;
    private final String branch;

    public GitRepositoryValidationStep(String url, String username, String password, 
                                        String gitRepositoryPath, String gitDirectoryId, String branch) {
        this.url = url;
        this.username = username;
        this.password = password;
        this.gitRepositoryPath = gitRepositoryPath;
        this.gitDirectoryId = gitDirectoryId;
        this.branch = branch != null && !branch.isEmpty() ? branch : "main";
    }

    @Override
    public String getDescription() {
        return "Validating Git Repository";
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        ctx.logOutput("Starting Git repository validation");
        
        // Log the authenticated XLD user who initiated this import
        // This provides traceability similar to xl-cli's user context handling
        logAuthenticatedUser(ctx);
        
        ctx.logOutput("Git repository path: " + gitRepositoryPath);
        ctx.logOutput("Git directory ID: " + gitDirectoryId);
        ctx.logOutput("Git URL: " + url);
        ctx.logOutput("Branch: " + branch);

        // Get Git username from credentials passed by delegate
        String gitSourceUsername = username != null ? username : "unknown";
        ctx.logOutput("Git Source User: " + gitSourceUsername);

        // Generate deterministic folder name: Folder_{hash}_username_reponame_branch
        // Hash is based on gitDirectoryId so each directory always uses the same folder
        String workspaceFolderName = YamlHandler.createGitRepoFolderName(gitDirectoryId, gitSourceUsername, url, branch);
        
        // Use work/git_repos under server home directory
        String serverHome = System.getProperty("user.dir");
        File gitReposDir = new File(serverHome, "work" + File.separator + "git_repos");
        File localRepoDir = new File(gitReposDir, workspaceFolderName);
        
        ctx.logOutput("Server home: " + serverHome);
        ctx.logOutput("Git repos directory: " + gitReposDir.getAbsolutePath());
        ctx.logOutput("Workspace folder name: " + workspaceFolderName);
        ctx.logOutput("Local repository directory: " + localRepoDir.getAbsolutePath());
        ctx.logOutput("Sparse checkout path: " + gitRepositoryPath);

        // Clean up any old folders for the same gitDirectoryId but different branch
        // This ensures only one folder per gitDirectory exists at any time
        String idHash = YamlHandler.generateShortHash(gitDirectoryId);
        cleanupOldFoldersForSameGitDirectory(gitReposDir, idHash, workspaceFolderName, ctx);

        UsernamePasswordCredentialsProvider credentialsProvider = null;
        if (username != null && !username.isEmpty() && password != null && !password.isEmpty()) {
            credentialsProvider = new UsernamePasswordCredentialsProvider(username, password);
        }

        if (!verifyPathExistsInRemote(url, branch, gitRepositoryPath, credentialsProvider, ctx)) {
            return StepExitCode.FAIL;
        }

        Git git = null;
        try {
            if (localRepoDir.exists() && new File(localRepoDir, ".git").exists()) {
                ctx.logOutput("Repository already exists, performing git pull on sparse checkout...");
                git = Git.open(localRepoDir);

                PullResult pullResult;
                if (credentialsProvider != null) {
                    pullResult = git.pull()
                        .setCredentialsProvider(credentialsProvider)
                        .setRemoteBranchName(branch)
                        .call();
                } else {
                    pullResult = git.pull()
                        .setRemoteBranchName(branch)
                        .call();
                }

                ctx.logOutput("Pull result: " + (pullResult.isSuccessful() ? "SUCCESS" : "FAILED"));
            } else {
                ctx.logOutput("Performing sparse clone (only " + gitRepositoryPath + ")...");
                localRepoDir.mkdirs();

                git = Git.init()
                    .setDirectory(localRepoDir)
                    .call();

                File sparseCheckoutFile = new File(localRepoDir, ".git/info/sparse-checkout");
                sparseCheckoutFile.getParentFile().mkdirs();
                Files.write(sparseCheckoutFile.toPath(), (gitRepositoryPath + "/*\n").getBytes());

                git.getRepository().getConfig().setBoolean("core", null, "sparseCheckout", true);
                git.getRepository().getConfig().save();

                git.remoteAdd()
                    .setName("origin")
                    .setUri(new org.eclipse.jgit.transport.URIish(url))
                    .call();

                ctx.logOutput("Fetching from remote...");

                if (credentialsProvider != null) {
                    git.fetch()
                        .setRemote("origin")
                        .setRefSpecs("+refs/heads/" + branch + ":refs/remotes/origin/" + branch)
                        .setCredentialsProvider(credentialsProvider)
                        .call();
                } else {
                    git.fetch()
                        .setRemote("origin")
                        .setRefSpecs("+refs/heads/" + branch + ":refs/remotes/origin/" + branch)
                        .call();
                }

                ctx.logOutput("Checking out branch: " + branch);
                git.checkout()
                    .setName(branch)
                    .setStartPoint("origin/" + branch)
                    .setCreateBranch(true)
                    .call();

                ctx.logOutput("Sparse clone completed successfully (only " + gitRepositoryPath + " directory)");
            }

            Path directoryPath = Paths.get(localRepoDir.getAbsolutePath(), gitRepositoryPath);

            if (!Files.exists(directoryPath)) {
                ctx.logError("Directory does not exist: " + directoryPath);
                return StepExitCode.FAIL;
            }

            if (!Files.isDirectory(directoryPath)) {
                ctx.logError("Path is not a directory: " + directoryPath);
                return StepExitCode.FAIL;
            }

            ctx.logOutput("Repository validation complete. Directory ready: " + directoryPath);
            ctx.setAttribute(GitOpsExecutionAttributes.LOCAL_REPO_DIR, localRepoDir.getAbsolutePath());
            ctx.setAttribute(GitOpsExecutionAttributes.TARGET_DIRECTORY, directoryPath.toString());
            ctx.setAttribute(GitOpsExecutionAttributes.GIT_REMOTE_URL, url);
            
            // Extract SCM traceability data from the Git repository
            SCMInfo scmData = SCMInfoExtractor.extractSCMInfo(localRepoDir, "");
            if (scmData != null) {
                ctx.setAttribute(GitOpsExecutionAttributes.SCM_TRACEABILITY_DATA, scmData);
                ctx.logOutput("SCM Traceability: commit=" + scmData.getCommit().substring(0, Math.min(7, scmData.getCommit().length())) + 
                             ", author=" + scmData.getAuthor());
            } else {
                ctx.logOutput("SCM traceability data could not be extracted from repository");
            }
            
            return StepExitCode.SUCCESS;

        } catch (GitAPIException e) {
            ctx.logError("Git operation failed: " + e.getMessage(), e);
            return StepExitCode.FAIL;
        } finally {
            if (git != null) {
                git.close();
            }
        }
    }

    @Override
    public int getOrder() {
        return 10;
    }
    /**
     * Cleans up old folders for the same gitDirectoryId but with a different branch.
     * This ensures only one folder per gitDirectory exists at any time.
     * 
     * @param gitReposDir The parent directory containing all git repo folders
     * @param idHash The hash of the gitDirectoryId (first part after "Folder_")
     * @param currentFolderName The folder name for the current branch (to keep)
     * @param ctx The execution context for logging
     */
    private void cleanupOldFoldersForSameGitDirectory(File gitReposDir, String idHash, 
                                                       String currentFolderName, ExecutionContext ctx) {
        if (!gitReposDir.exists() || !gitReposDir.isDirectory()) {
            return;
        }
        
        String folderPrefix = "Folder_" + idHash + "_";
        File[] existingFolders = gitReposDir.listFiles();
        
        if (existingFolders == null) {
            return;
        }
        
        for (File folder : existingFolders) {
            if (folder.isDirectory() && folder.getName().startsWith(folderPrefix)) {
                // This folder belongs to the same gitDirectoryId
                if (!folder.getName().equals(currentFolderName)) {
                    // This is an old folder for a different branch - delete it
                    ctx.logOutput("Found old folder for different branch: " + folder.getName());
                    ctx.logOutput("Deleting old folder to maintain single folder per gitDirectory...");
                    try {
                        deleteDirectoryRecursively(folder.toPath());
                        ctx.logOutput("Successfully deleted old folder: " + folder.getName());
                    } catch (Exception e) {
                        ctx.logError("Warning: Failed to delete old folder " + folder.getName() + ": " + e.getMessage());
                    }
                }
            }
        }
    }
    
    /**
     * Recursively deletes a directory and all its contents.
     * Handles Windows-specific issues with .git folders by making files writable before deletion.
     * 
     * @param path The path to delete
     */
    private void deleteDirectoryRecursively(Path path) throws Exception {
        if (!Files.exists(path)) {
            return;
        }
        
        // Walk the file tree in reverse order (depth first) to delete files before directories
        java.util.List<Path> pathsToDelete = new java.util.ArrayList<>();
        Files.walk(path)
            .sorted(java.util.Comparator.reverseOrder())
            .forEach(pathsToDelete::add);
        
        for (Path p : pathsToDelete) {
            try {
                // On Windows, .git folder files may be read-only - make them writable first
                File file = p.toFile();
                if (file.exists() && !file.canWrite()) {
                    file.setWritable(true);
                }
                Files.deleteIfExists(p);
            } catch (Exception e) {
                // Try one more time after a small delay (file locks)
                try {
                    Thread.sleep(50);
                    File file = p.toFile();
                    file.setWritable(true);
                    file.setReadable(true);
                    if (file.isDirectory()) {
                        file.delete();
                    } else {
                        Files.deleteIfExists(p);
                    }
                } catch (Exception e2) {
                    throw new Exception("Failed to delete: " + p, e2);
                }
            }
        }
    }
    private boolean verifyPathExistsInRemote(String url, String branch, String path,
                                              UsernamePasswordCredentialsProvider credentialsProvider,
                                              ExecutionContext ctx) {
        File tempDir = null;
        Git git = null;
        try {
            tempDir = Files.createTempDirectory("git_path_verify_").toFile();

            var cloneCommand = Git.cloneRepository()
                .setURI(url)
                .setDirectory(tempDir)
                .setBranch(branch)
                .setDepth(1)
                .setNoCheckout(true);

            if (credentialsProvider != null) {
                cloneCommand.setCredentialsProvider(credentialsProvider);
            }

            git = cloneCommand.call();
            Repository repository = git.getRepository();

            ObjectId branchId = repository.resolve("refs/heads/" + branch);
            if (branchId == null) {
                branchId = repository.resolve("origin/" + branch);
            }
            if (branchId == null) {
                ctx.logError("Branch '" + branch + "' does not exist in the remote repository.");
                return false;
            }

            try (RevWalk revWalk = new RevWalk(repository)) {
                RevCommit commit = revWalk.parseCommit(branchId);
                RevTree tree = commit.getTree();

                try (TreeWalk treeWalk = TreeWalk.forPath(repository, path, tree)) {
                    if (treeWalk == null) {
                        ctx.logError("Path '" + path + "' does not exist in branch '" + branch + "'.");
                        return false;
                    }
                    return true;
                }
            }

        } catch (Exception e) {
            String errorMessage = e.getMessage();
            
            // Check if this is a branch-not-found error
            if (errorMessage != null && (
                    errorMessage.contains("Remote branch") && errorMessage.contains("not found") ||
                    errorMessage.contains("not advertise Ref for branch") ||
                    errorMessage.contains("cannot be resolved") ||
                    errorMessage.contains("Ref") && errorMessage.contains("not exist"))) {
                ctx.logError("Branch '" + branch + "' does not exist in the remote repository.");
                return false;
            }
            
            // For other errors (network issues, permissions), log and proceed
            ctx.logError("Warning: " + errorMessage);
            return true;
        } finally {
            if (git != null) {
                git.close();
            }
            if (tempDir != null && tempDir.exists()) {
                deleteDirectoryRecursively(tempDir);
            }
        }
    }

    private void deleteDirectoryRecursively(File directory) {
        if (directory.isDirectory()) {
            File[] files = directory.listFiles();
            if (files != null) {
                for (File file : files) {
                    deleteDirectoryRecursively(file);
                }
            }
        }
        directory.delete();
    }

    /**
     * Logs the authenticated XLD user who initiated this import task.
     * This provides traceability similar to xl-cli's user context handling.
     * 
     * When xl-cli runs 'xl apply', it authenticates with the server using credentials
     * and all operations are performed under that user's permissions. The gitops-plugin
     * works similarly - the user who triggers the import from the UI is the authenticated
     * user, and all CI creation/updates honour that user's permissions.
     * 
     * @param ctx The execution context
     */
    private void logAuthenticatedUser(ExecutionContext ctx) {
        try {
            Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
            if (authentication != null && authentication.getName() != null) {
                String xldUser = authentication.getName();
                ctx.logOutput("XLD Authenticated User: " + xldUser);
                ctx.logOutput("All CI changes will be performed under this user's permissions");
            } else {
                ctx.logOutput("XLD Authenticated User: unknown (no security context available)");
            }
        } catch (Exception e) {
            // Don't fail the step if we can't get the user info
            ctx.logOutput("XLD Authenticated User: could not determine (" + e.getMessage() + ")");
        }
    }
}
