package ai.digital.deploy.gitops.util;

import com.xebialabs.deployit.engine.api.dto.XLDAsCodeResult;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Utility class to parse and log AsCodeResponse changes and errors from XLDAsCodeResult.
 * Parses the JSON content and formats Created/Updated/Deleted messages like the xl CLI.
 * Also parses and logs validation, permission, document, and generic errors.
 */
public class AsCodeResponseLogger {

    // Patterns to extract arrays from JSON for changes
    private static final Pattern CREATED_PATTERN = Pattern.compile("\"created\"\\s*:\\s*\\[([^\\]]*)]");
    private static final Pattern UPDATED_PATTERN = Pattern.compile("\"updated\"\\s*:\\s*\\[([^\\]]*)]");
    private static final Pattern DELETED_PATTERN = Pattern.compile("\"deleted\"\\s*:\\s*\\[([^\\]]*)]");
    private static final Pattern KIND_PATTERN = Pattern.compile("\"kind\"\\s*:\\s*\"([^\"]+)\"");
    private static final Pattern ID_PATTERN = Pattern.compile("\"([^\"]+)\"");
    
    // Patterns for error parsing
    private static final Pattern ERRORS_BLOCK_PATTERN = Pattern.compile("\"errors\"\\s*:\\s*\\{([^}]+(?:\\{[^}]*\\}[^}]*)*)\\}");
    private static final Pattern VALIDATION_ARRAY_PATTERN = Pattern.compile("\"validation\"\\s*:\\s*\\[([^\\]]*(?:\\{[^}]*\\}[^\\]]*)*)\\]");
    private static final Pattern PERMISSION_ARRAY_PATTERN = Pattern.compile("\"permission\"\\s*:\\s*\\[([^\\]]*(?:\\{[^}]*\\}[^\\]]*)*)\\]");
    private static final Pattern DOCUMENT_PATTERN = Pattern.compile("\"document\"\\s*:\\s*\\{([^}]*)\\}");
    private static final Pattern GENERIC_PATTERN = Pattern.compile("\"generic\"\\s*:\\s*\"([^\"]+)\"");
    
    // Patterns for validation error fields
    private static final Pattern CI_ID_PATTERN = Pattern.compile("\"ciId\"\\s*:\\s*\"([^\"]+)\"");
    private static final Pattern PROPERTY_NAME_PATTERN = Pattern.compile("\"propertyName\"\\s*:\\s*\"([^\"]+)\"");
    private static final Pattern MESSAGE_PATTERN = Pattern.compile("\"message\"\\s*:\\s*\"([^\"]+)\"");
    private static final Pattern PERMISSION_PATTERN = Pattern.compile("\"permission\"\\s*:\\s*\"([^\"]+)\"");
    private static final Pattern FIELD_PATTERN = Pattern.compile("\"field\"\\s*:\\s*\"([^\"]+)\"");
    private static final Pattern PROBLEM_PATTERN = Pattern.compile("\"problem\"\\s*:\\s*\"([^\"]+)\"");

    /**
     * Logs the changes from an XLDAsCodeResult to the execution context.
     * Format matches the xl CLI output:
     *   Created CI Infrastructure/devDir/infraTest
     *   Updated CI Infrastructure/devDir
     *
     * @param result The result from the XLDAsCodeService.apply() call
     * @param ctx The execution context for logging
     */
    public static void logChanges(XLDAsCodeResult result, ExecutionContext ctx) {
        if (result == null || result.getContent() == null) {
            return;
        }

        String content = result.getContent();
        
        // Try to parse as JSON
        if (!content.startsWith("{")) {
            // Not JSON format, skip parsing
            return;
        }

        try {
            // Extract kind (usually "CI")
            String kind = extractKind(content);
            
            // Extract and log created items
            List<String> created = extractIds(content, CREATED_PATTERN);
            for (String id : created) {
                ctx.logOutput("    Created " + kind + " " + id);
            }
            
            // Extract and log updated items
            List<String> updated = extractIds(content, UPDATED_PATTERN);
            for (String id : updated) {
                ctx.logOutput("    Updated " + kind + " " + id);
            }
            
            // Extract and log deleted items
            List<String> deleted = extractIds(content, DELETED_PATTERN);
            for (String id : deleted) {
                ctx.logOutput("    Deleted " + kind + " " + id);
            }
            
            // Log any errors from the response
            logErrors(content, ctx);
            
        } catch (Exception e) {
            // Silently ignore parsing errors - the main result is already logged
        }
    }
    
    /**
     * Logs errors from the AsCodeResponse JSON content.
     * Parses and formats validation, permission, document, and generic errors.
     */
    private static void logErrors(String content, ExecutionContext ctx) {
        // Check if there's an errors block
        Matcher errorsMatcher = ERRORS_BLOCK_PATTERN.matcher(content);
        if (!errorsMatcher.find()) {
            return;
        }
        
        String errorsBlock = errorsMatcher.group(0);
        
        // Parse validation errors: { ciId, propertyName, message }
        Matcher validationMatcher = VALIDATION_ARRAY_PATTERN.matcher(errorsBlock);
        if (validationMatcher.find()) {
            String validationArray = validationMatcher.group(1);
            // Split by }, { to get individual error objects
            String[] validationErrors = validationArray.split("\\}\\s*,\\s*\\{");
            for (String error : validationErrors) {
                String ciId = extractField(error, CI_ID_PATTERN);
                String propertyName = extractField(error, PROPERTY_NAME_PATTERN);
                String message = extractField(error, MESSAGE_PATTERN);
                if (ciId != null || message != null) {
                    ctx.logOutput("    Validation Error: " + 
                        (ciId != null ? ciId : "") + 
                        (propertyName != null ? " [" + propertyName + "]" : "") + 
                        (message != null ? " - " + message : ""));
                }
            }
        }
        
        // Parse permission errors: { ciId, permission }
        Matcher permissionMatcher = PERMISSION_ARRAY_PATTERN.matcher(errorsBlock);
        if (permissionMatcher.find()) {
            String permissionArray = permissionMatcher.group(1);
            String[] permissionErrors = permissionArray.split("\\}\\s*,\\s*\\{");
            for (String error : permissionErrors) {
                String ciId = extractField(error, CI_ID_PATTERN);
                String permission = extractField(error, PERMISSION_PATTERN);
                if (ciId != null || permission != null) {
                    ctx.logOutput("    Permission Error: " + 
                        (ciId != null ? ciId : "") + 
                        (permission != null ? " - Missing permission: " + permission : ""));
                }
            }
        }
        
        // Parse document error: { field, problem }
        Matcher documentMatcher = DOCUMENT_PATTERN.matcher(errorsBlock);
        if (documentMatcher.find()) {
            String documentError = documentMatcher.group(1);
            String field = extractField(documentError, FIELD_PATTERN);
            String problem = extractField(documentError, PROBLEM_PATTERN);
            if (field != null || problem != null) {
                ctx.logOutput("    Document Error: " + 
                    (field != null ? "Field '" + field + "'" : "") + 
                    (problem != null ? " - " + problem : ""));
            }
        }
        
        // Parse generic error: string
        Matcher genericMatcher = GENERIC_PATTERN.matcher(errorsBlock);
        if (genericMatcher.find()) {
            String genericError = genericMatcher.group(1);
            if (genericError != null && !genericError.isEmpty()) {
                ctx.logOutput("    Error: " + genericError);
            }
        }
    }
    
    /**
     * Extracts a field value from JSON using the given pattern.
     */
    private static String extractField(String json, Pattern pattern) {
        Matcher matcher = pattern.matcher(json);
        if (matcher.find()) {
            return matcher.group(1);
        }
        return null;
    }

    /**
     * Extracts the kind value from the JSON content.
     */
    private static String extractKind(String content) {
        Matcher matcher = KIND_PATTERN.matcher(content);
        if (matcher.find()) {
            return matcher.group(1);
        }
        return "CI"; // Default to CI
    }

    /**
     * Extracts IDs from a JSON array matched by the given pattern.
     */
    private static List<String> extractIds(String content, Pattern arrayPattern) {
        List<String> ids = new ArrayList<>();
        
        Matcher arrayMatcher = arrayPattern.matcher(content);
        if (arrayMatcher.find()) {
            String arrayContent = arrayMatcher.group(1);
            if (arrayContent != null && !arrayContent.trim().isEmpty()) {
                Matcher idMatcher = ID_PATTERN.matcher(arrayContent);
                while (idMatcher.find()) {
                    ids.add(idMatcher.group(1));
                }
            }
        }
        
        return ids;
    }

    /**
     * Returns true if the result contains any changes (created, updated, or deleted).
     */
    public static boolean hasChanges(XLDAsCodeResult result) {
        if (result == null || result.getContent() == null) {
            return false;
        }

        String content = result.getContent();
        if (!content.startsWith("{")) {
            return false;
        }

        List<String> created = extractIds(content, CREATED_PATTERN);
        List<String> updated = extractIds(content, UPDATED_PATTERN);
        List<String> deleted = extractIds(content, DELETED_PATTERN);

        return !created.isEmpty() || !updated.isEmpty() || !deleted.isEmpty();
    }
    
    /**
     * Returns true if the result contains any errors (validation, permission, document, or generic).
     */
    public static boolean hasErrors(XLDAsCodeResult result) {
        if (result == null || result.getContent() == null) {
            return false;
        }

        String content = result.getContent();
        if (!content.startsWith("{")) {
            return false;
        }

        return ERRORS_BLOCK_PATTERN.matcher(content).find();
    }
}
