package ai.digital.deploy.gitops.step;

import ai.digital.deploy.gitops.service.GitOpsServiceHolder;
import com.xebialabs.deployit.engine.api.ServiceHolder;
import com.xebialabs.deployit.engine.api.XLDAsCodeService;
import com.xebialabs.deployit.engine.api.dto.XLDAsCodeResult;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;

public class GitApplyYamlStep implements Step {

    private final String gitDirectoryId;

    public GitApplyYamlStep(String gitDirectoryId) {
        this.gitDirectoryId = gitDirectoryId;
    }

    @Override
    public String getDescription() {
        return "Importing Configuration Items";
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        Object yamlFilesAttr = ctx.getAttribute(GitOpsExecutionAttributes.YAML_FILES);
        if (!(yamlFilesAttr instanceof List)) {
            ctx.logError("YAML file list not found in context. Did the YAML validation step run?");
            return StepExitCode.FAIL;
        }

        @SuppressWarnings("unchecked")
        List<String> yamlFiles = (List<String>) yamlFilesAttr;
        if (yamlFiles.isEmpty()) {
            ctx.logError("No YAML files to process.");
            return StepExitCode.FAIL;
        }

        XLDAsCodeService xldAsCodeService = ServiceHolder.getXLDAsCodeService();
        GitOpsServiceHolder.GitOpsServiceAdapter adapter = null;

        if (xldAsCodeService == null) {
            ctx.logOutput("XLDAsCodeService not available. Falling back to adapter.");
            adapter = GitOpsServiceHolder.getGitOpsService();
        }

        int successCount = 0;
        int failureCount = 0;

        for (String yamlFilePath : yamlFiles) {
            Path yamlFile = Paths.get(yamlFilePath);
            ctx.logOutput("Processing YAML file: " + yamlFile);

            try {
                String yamlContent = new String(Files.readAllBytes(yamlFile));

                if (yamlContent.trim().isEmpty()) {
                    ctx.logOutput("  Skipped (empty file)");
                    continue;
                }

                XLDAsCodeResult result = (xldAsCodeService != null)
                    ? xldAsCodeService.apply(yamlContent)
                    : adapter.apply(yamlContent);

                if (result.isSuccess()) {
                    ctx.logOutput("  Success: " + yamlFile.getFileName());
                    if (result.getMessage() != null) {
                        ctx.logOutput("    " + result.getMessage());
                    }
                    successCount++;
                } else {
                    ctx.logError("  Failed: " + yamlFile.getFileName());
                    ctx.logError("  Reason: " + result.getMessage());
                    if (result.getError() != null) {
                        ctx.logError("  Error: " + result.getError());
                    }
                    failureCount++;
                }

            } catch (Exception e) {
                ctx.logError("  Exception processing file: " + e.getMessage());
                failureCount++;
            }
        }

        ctx.logOutput("Apply YAML completed: " + successCount + " successful, " + failureCount + " failed");
        return successCount > 0 ? StepExitCode.SUCCESS : StepExitCode.FAIL;
    }

    @Override
    public int getOrder() {
        return 30;
    }
}
