package ai.digital.deploy.gitops;

import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.CloneCommand;
import org.eclipse.jgit.api.ListBranchCommand;
import org.eclipse.jgit.api.PullCommand;
import org.eclipse.jgit.api.PullResult;
import org.eclipse.jgit.api.ResetCommand;
import org.eclipse.jgit.api.Status;
import org.eclipse.jgit.api.StatusCommand;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.api.MergeResult;
import org.eclipse.jgit.merge.MergeStrategy;
import org.eclipse.jgit.transport.CredentialsProvider;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import ai.digital.deploy.gitops.util.YamlHandler;
import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;

public class DefaultGitOpsService implements GitOpsService {
    private static final Logger logger = LoggerFactory.getLogger(DefaultGitOpsService.class);
    
    /**
     * Creates a CredentialsProvider from GitUserCredentials if available.
     */
    private CredentialsProvider createCredentialsProvider(GitUserCredentials credentials) {
        if (credentials != null && credentials.hasCredentials()) {
            return new UsernamePasswordCredentialsProvider(
                credentials.getUsername(), 
                credentials.getPassword()
            );
        }
        return null;
    }
    
    /**
     * Performs Git lsRemoteRepository operation with given parameters.
     */


    @Override
    public void testConnection(String url, String branch, GitUserCredentials credentials) {
        logger.info("Testing Git connection to: {}", url);
        
        try {
            CredentialsProvider credentialsProvider = createCredentialsProvider(credentials);
            
            // Test connection by listing remote branches
            java.util.Collection<Ref> refs;
            try {
                refs = listRemoteBranches(url, credentialsProvider);
            } catch (Exception listError) {
                throw new RuntimeException("Failed to list remote branches: " + listError.getMessage(), listError);
            }
            
            if (refs.isEmpty()) {
                throw new RuntimeException("No branches found in repository: " + url);
            }
            
            // Validate branch exists if specified
            validateBranchExists(branch, refs, url);
            
            logger.info("Git connection test successful. Repository has {} branches", refs.size());
            
        } catch (Exception e) {
            logger.error("Git connection test failed for URL: {}", url, e);
            if (!(e instanceof RuntimeException)) {
                throw new RuntimeException("Failed to connect to Git repository: " + e.getMessage(), e);
            }
            throw e;
        }
    }
    
    /**
     * Validates that the specified branch exists in the remote repository.
     * Skips validation for default branches ('main' and 'master') and null/empty branch names.
     * 
     * @param branch the branch name to validate (can be null or empty)
     * @param refs collection of remote references from the repository
     * @param url the repository URL (used for error messaging)
     * @throws RuntimeException if the specified branch does not exist in the repository
     */
    private void validateBranchExists(String branch, java.util.Collection<Ref> refs, String url) {
        if (branch != null && !branch.isEmpty() && !branch.equals("main") && !branch.equals("master")) {
            boolean branchExists = refs.stream()
                .anyMatch(ref -> ref.getName().endsWith("/" + branch));
            
            if (!branchExists) {
                logger.error("Branch '{}' not found in repository. Available branches: {}", 
                    branch, refs.stream().map(ref -> ref.getName()).toArray());
                throw new RuntimeException("Branch '" + branch + "' does not exist in repository: " + url);
            }
        }
    }

    @Override
    public String cloneRepository(String url, String branch, GitUserCredentials credentials, String targetDirectory) {
        logger.info("Cloning Git repository from: {} to: {}", url, targetDirectory);
        
        try {
            // Validate connection (credentials and branch) using existing testConnection method
            testConnection(url, branch, credentials);
            
            CredentialsProvider credentialsProvider = createCredentialsProvider(credentials);

            // Use consistent org-prefixed repository name for directory naming
            String orgPrefixedRepoName = YamlHandler.createOrgPrefixedRepositoryName(url);
            logger.debug("Using org-prefixed repository name: {}", orgPrefixedRepoName);
            
            // Create final target directory: targetDirectory/orgPrefixedRepoName
            String finalTargetDirectory = createUniqueTargetDirectory(targetDirectory, orgPrefixedRepoName, url);
            File targetDir = new File(finalTargetDirectory);
            
            ensureDirectoryExists(targetDir);
            
            // Clone command setup
            CloneCommand cloneCommand = Git.cloneRepository()
                .setURI(url)
                .setDirectory(targetDir)
                .setCredentialsProvider(credentialsProvider);
                
            if (branch != null && !branch.isEmpty()) {
                cloneCommand.setBranch(branch);
                logger.debug("Cloning specific branch: {}", branch);
            }

            try (Git git = cloneCommand.call()) {
                logger.info("Git repository cloned successfully to: {}", finalTargetDirectory);
                
                // Log information about the cloned repository
                String currentBranch = git.getRepository().getBranch();
                logger.info("Currently on branch: {}", currentBranch);
                
                // Log repository information
                File gitDir = git.getRepository().getDirectory();
                logger.debug("Git directory: {}", gitDir.getAbsolutePath());
                
                // Return the actual directory where repository was cloned
                return finalTargetDirectory;
            }
            
        } catch (Exception e) {
            // Error handling - wrap in RuntimeException with meaningful message
            logger.error("Git clone failed for URL: {} to directory: {}", url, targetDirectory, e);
            throw new RuntimeException("Failed to clone Git repository: " + e.getMessage(), e);
        }
    }
    

    
    /**
     * Extracts the actual folder name from the cloned directory path.
     * This is more reliable than URL parsing since it uses the actual created folder name.
     * Examples:
     * - /path/to/clone/repo -> repo  
     * - /path/to/clone/repo_1 -> repo_1
     * - C:\temp\my-repo -> my-repo
     */
    public static String extractFolderNameFromPath(String clonedPath) {
        if (clonedPath == null || clonedPath.trim().isEmpty()) {
            return "unknown";
        }
        
        try {
            File clonedDir = new File(clonedPath);
            String folderName = clonedDir.getName();
            
            // Return the folder name, or fallback if empty
            return (folderName != null && !folderName.trim().isEmpty()) ? folderName : "unknown";
            
        } catch (Exception e) {
            // Fallback to manual parsing if File constructor fails
            String normalizedPath = clonedPath.replace('\\', '/');
            if (normalizedPath.endsWith("/")) {
                normalizedPath = normalizedPath.substring(0, normalizedPath.length() - 1);
            }
            
            int lastSlash = normalizedPath.lastIndexOf('/');
            if (lastSlash >= 0 && lastSlash < normalizedPath.length() - 1) {
                return normalizedPath.substring(lastSlash + 1);
            }
            
            return "unknown";
        }
    }
    
    /**
     * Creates a unique target directory using the provided repository name.
     * If directory already exists and is not empty, falls back to timestamp-based naming.
     * Examples:
     * - /path/to/clone + repo_orgname -> /path/to/clone/repo_orgname
     * - /path/to/clone + repo_orgname (if exists and not empty) -> /path/to/clone/repo_orgname_timestamp
     */
    private String createUniqueTargetDirectory(String baseDirectory, String repoName, String repoUrl) {
        // Normalize the base directory path to handle cross-platform issues
        // If baseDirectory is a Windows absolute path (e.g., C:\path) but we're on Linux,
        // treat it as a relative path from current working directory
        File baseDir = normalizeBasePath(baseDirectory);
        
        // Create base directory if it doesn't exist
        if (!baseDir.exists()) {
            boolean created = baseDir.mkdirs();
            if (!created) {
                logger.warn("Failed to create base directory: {}", baseDirectory);
            }
        }
        
        File targetDir = new File(baseDir, repoName);
        
        // If directory doesn't exist, use it
        if (!targetDir.exists()) {
            return targetDir.getAbsolutePath();
        }
        
        // If directory exists and is empty, use it
        if (targetDir.isDirectory() && targetDir.list() != null && targetDir.list().length == 0) {
            return targetDir.getAbsolutePath();
        }
        
        // Directory exists and is not empty, fall back to timestamp-based naming
        String timestamp = String.valueOf(System.currentTimeMillis());
        String timestampName = repoName + "_" + timestamp;
        File timestampDir = new File(baseDir, timestampName);
        logger.info("Directory {} exists and is not empty, using timestamp-based name: {}", repoName, timestampName);
        return timestampDir.getAbsolutePath();
    }

    @Override
    public String cloneRepository(ConfigurationItem gitSource, String targetDirectory) {
        if (gitSource == null) {
            throw new IllegalArgumentException("Git source cannot be null");
        }

        String url = (String) gitSource.getProperty("url");
        String branch = (String) gitSource.getProperty("branch");
        
        // Extract credentials from ConfigurationItem
        GitUserCredentials credentials = null;
        try {
            String username = (String) gitSource.getProperty("username");
            String password = (String) gitSource.getProperty("password");
            
            if (username != null && !username.trim().isEmpty() && 
                password != null && !password.trim().isEmpty()) {
                credentials = new GitUserCredentials(username, password);
            }
        } catch (Exception e) {
            // Ignore credential extraction errors
        }
        
        return cloneRepository(url, branch, credentials, targetDirectory);
    }

    @Override
    public String pullRepository(String repositoryPath, GitUserCredentials credentials) {
        logger.info("Pulling Git repository at: {}", repositoryPath);
        
        try {
            validateRepositoryDirectory(repositoryPath);
            CredentialsProvider credentialsProvider = createCredentialsProvider(credentials);
            

            
            try (Git git = Git.open(new File(repositoryPath))) {
                String currentBranch = git.getRepository().getBranch();
                logger.info("Pulling updates for branch: {}", currentBranch);
                
                PullCommand pullCommand = git.pull();
                if (credentialsProvider != null) {
                    pullCommand.setCredentialsProvider(credentialsProvider);
                }
                
                PullResult result = pullCommand.call();
                return handlePullResult(result, repositoryPath, git, credentialsProvider);
            }
            
        } catch (Exception e) {
            logger.error("Git pull failed for repository: {}", repositoryPath, e);
            throw new RuntimeException("Failed to pull Git repository: " + e.getMessage(), e);
        }
    }
    
    private void validateRepositoryDirectory(String repositoryPath) {
        File repoDir = new File(repositoryPath);
        if (!repoDir.exists() || !repoDir.isDirectory()) {
            throw new RuntimeException("Repository directory does not exist: " + repositoryPath);
        }
        
        if (!isGitRepository(repositoryPath)) {
            throw new RuntimeException("Directory is not a Git repository: " + repositoryPath);
        }
    }
    
    private String handlePullResult(PullResult result, String repositoryPath, Git git, CredentialsProvider credentialsProvider) throws Exception {
        if (result.isSuccessful()) {
            logger.info("Git pull completed successfully for: {}", repositoryPath);
            
            // Log merge details if available
            if (result.getMergeResult() != null) {
                org.eclipse.jgit.api.MergeResult mergeResult = result.getMergeResult();
                logger.info("Merge status: {}", mergeResult.getMergeStatus());
                if (mergeResult.getConflicts() != null && !mergeResult.getConflicts().isEmpty()) {
                    logger.warn("Pull successful but had conflicts that were auto-resolved: {}", 
                        mergeResult.getConflicts().keySet());
                }
            }
            
            return repositoryPath;
            
        } else {
            // Pull failed - check for merge conflicts
            org.eclipse.jgit.api.MergeResult mergeResult = result.getMergeResult();
            if (mergeResult != null) {
                org.eclipse.jgit.api.MergeResult.MergeStatus mergeStatus = mergeResult.getMergeStatus();
                logger.error("Pull failed with merge status: {}", mergeStatus);
                
                if (mergeStatus == org.eclipse.jgit.api.MergeResult.MergeStatus.CONFLICTING) {
                    return handleMergeConflicts(git, repositoryPath, mergeResult, credentialsProvider);
                } else {
                    String errorMsg = "Git pull failed due to merge issues: " + mergeStatus;
                    logger.error(errorMsg);
                    throw new RuntimeException(errorMsg);
                }
            } else {
                String errorMsg = "Git pull failed: " + result.toString();
                logger.error(errorMsg);
                throw new RuntimeException(errorMsg);
            }
        }
    }

    /**
     * Determines the actual repository path that would be used for cloning,
     * taking into account collision resolution logic.
     */
    public String determineRepositoryPath(String url, String targetDirectory) {
        String orgPrefixedRepoName = YamlHandler.createOrgPrefixedRepositoryName(url);
        return createUniqueTargetDirectory(targetDirectory, orgPrefixedRepoName, url);
    }

    @Override
    public boolean isGitRepository(String directoryPath) {
        try {
            File repoDir = new File(directoryPath);
            if (!repoDir.exists() || !repoDir.isDirectory()) {
                return false;
            }
            
            // Check if .git directory exists
            File gitDir = new File(repoDir, ".git");
            return gitDir.exists() && gitDir.isDirectory();
            
        } catch (Exception e) {
            logger.debug("Error checking if directory is Git repository: {}", directoryPath, e);
            return false;
        }
    }
    
    /**
     * Handle merge conflicts during pull operation.
     * Provides multiple strategies for conflict resolution.
     */
    private String handleMergeConflicts(Git git, String repositoryPath, org.eclipse.jgit.api.MergeResult mergeResult, CredentialsProvider credentials) {
        logger.warn("Merge conflicts detected during pull operation");
        
        try {
            // Log conflicted files
            if (mergeResult.getConflicts() != null) {
                logger.warn("Conflicted files:");
                mergeResult.getConflicts().keySet().forEach(file -> 
                    logger.warn("  - {}", file));
            }
            
            // Check repository status to get more details
            Status status = git.status().call();
            if (status.hasUncommittedChanges()) {
                logger.warn("Repository has uncommitted changes:");
                status.getConflicting().forEach(file -> logger.warn("  CONFLICTING: {}", file));
                status.getModified().forEach(file -> logger.warn("  MODIFIED: {}", file));
                status.getAdded().forEach(file -> logger.warn("  ADDED: {}", file));
            }
            
            // Strategy 1: Try to abort merge and use "theirs" strategy (remote wins)
            logger.info("Attempting conflict resolution using 'theirs' strategy (remote wins)");
            
            // Reset to HEAD to abort the merge
            git.reset().setMode(ResetCommand.ResetType.HARD).setRef("HEAD").call();
            logger.info("Reset to HEAD completed");
            
            // Pull again with theirs strategy
            PullCommand retryPull = git.pull().setStrategy(MergeStrategy.THEIRS);
            if (credentials != null) {
                retryPull.setCredentialsProvider(credentials);
            }
            
            PullResult retryResult = retryPull.call();
            
            if (retryResult.isSuccessful()) {
                logger.info("Merge conflicts resolved using 'theirs' strategy");
                logger.warn("Local changes were overwritten by remote changes");
                return repositoryPath;
            } else {
                // Strategy 2: Force reset to remote HEAD
                logger.warn("'Theirs' strategy failed, attempting hard reset to remote");
                return performHardResetToRemote(git, repositoryPath, credentials);
            }
            
        } catch (Exception e) {
            logger.error("Failed to handle merge conflicts", e);
            // Strategy 3: Last resort - suggest manual intervention
            return handleConflictResolutionFailure(repositoryPath, e);
        }
    }
    
    /**
     * Perform hard reset to remote HEAD as last resort for conflicts.
     */
    private String performHardResetToRemote(Git git, String repositoryPath, CredentialsProvider credentials) {
        try {
            logger.warn("Performing hard reset to remote HEAD - all local changes will be lost");
            
            // Fetch latest changes
            if (credentials != null) {
                git.fetch().setCredentialsProvider(credentials).call();
            } else {
                git.fetch().call();
            }
            
            // Get current branch name
            String currentBranch = git.getRepository().getBranch();
            String remoteBranch = "origin/" + currentBranch;
            
            // Hard reset to remote branch
            git.reset()
                .setMode(ResetCommand.ResetType.HARD)
                .setRef(remoteBranch)
                .call();
                
            logger.info("Hard reset to {} completed", remoteBranch);
            logger.warn("All local changes and commits were discarded");
            
            return repositoryPath;
            
        } catch (Exception e) {
            logger.error("Hard reset to remote failed", e);
            throw new RuntimeException("Unable to resolve merge conflicts: " + e.getMessage(), e);
        }
    }
    
    /**
     * Handle the case where all conflict resolution strategies fail.
     */
    private String handleConflictResolutionFailure(String repositoryPath, Exception originalException) {
        String errorMessage = String.format(
            "Merge conflict resolution failed for repository: %s\n" +
            "Manual intervention required:\n" +
            "1. Navigate to: %s\n" +
            "2. Resolve conflicts manually using: git status, git add, git commit\n" +
            "3. Or reset repository: git reset --hard origin/<branch>\n" +
            "Original error: %s",
            repositoryPath, repositoryPath, originalException.getMessage()
        );
        
        logger.error(errorMessage);
        throw new RuntimeException(errorMessage, originalException);
    }
    
    // Helper methods for code simplification
    private java.util.Collection<Ref> listRemoteBranches(String url, CredentialsProvider credentialsProvider) throws Exception {
        return Git.lsRemoteRepository()
            .setRemote(url)
            .setCredentialsProvider(credentialsProvider)
            .setHeads(true)
            .call();
    }
    
    /**
     * Normalizes the base path to handle cross-platform path issues.
     * 
     * When a Windows absolute path (e.g., C:\XL\cloned) is used on a Linux/Unix system,
     * it should be treated as a relative path from the current working directory.
     * This prevents errors like "/opt/xebialabs/deploy-task-engine/C:\XL\cloned"
     * 
     * @param baseDirectory The base directory path (potentially Windows absolute path)
     * @return File object with properly normalized path
     */
    private File normalizeBasePath(String baseDirectory) {
        File basePath = new File(baseDirectory);
        
        // Check if we're on a Unix-like system and the path looks like a Windows absolute path
        if (File.separatorChar == '/' && isWindowsAbsolutePath(baseDirectory)) {
            // On Unix but given Windows path - treat it as relative path
            logger.warn("Detected Windows absolute path '{}' on Unix system. Treating as relative path.", baseDirectory);
            
            // Extract the meaningful part after drive letter
            // C:\XL\cloned -> XL/cloned
            String relativePath = baseDirectory.replaceFirst("^[A-Za-z]:[/\\\\]", "")
                                               .replace('\\', '/');
            basePath = new File(relativePath);
            logger.info("Normalized Windows path to relative path: {}", basePath.getAbsolutePath());
        }
        
        return basePath;
    }
    
    /**
     * Checks if the given path is a Windows absolute path (e.g., C:\path or C:/path)
     */
    private boolean isWindowsAbsolutePath(String path) {
        if (path == null || path.length() < 3) {
            return false;
        }
        // Check for pattern like "C:\" or "C:/"
        return path.matches("^[A-Za-z]:[/\\\\].*");
    }
    
    private void ensureDirectoryExists(Path directory) {
        try {
            if (!Files.exists(directory)) {
                Files.createDirectories(directory);
            }
        } catch (Exception e) {
            throw new RuntimeException("Failed to create directory: " + directory, e);
        }
    }
    
    private void ensureDirectoryExists(File directory) {
        if (!directory.exists()) {
            boolean created = directory.mkdirs();
            if (!created) {
                throw new RuntimeException("Failed to create target directory: " + directory.getAbsolutePath());
            }
            logger.debug("Created target directory: {}", directory.getAbsolutePath());
        }
    }
}