package ai.digital.deploy.gitops.step;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.rules.StepMetadata;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import ai.digital.deploy.gitops.DefaultGitOpsService;
import ai.digital.deploy.gitops.GitUserCredentials;
import ai.digital.deploy.gitops.service.GitOpsServiceHolder;
import ai.digital.deploy.gitops.util.YamlHandler;
import com.xebialabs.xlplatform.satellite.Satellite;
import com.xebialabs.xlplatform.satellite.SatelliteAware;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;

@StepMetadata(name = "gitClone")
public class GitCloneStep implements SatelliteAware {
    
    private ConfigurationItem gitSource;
    private String targetDirectory;
    
    // Default constructor required by XL Deploy rule scanning
    public GitCloneStep() {
    }
    
    public GitCloneStep(ConfigurationItem gitSource, String targetDirectory) {
        this.gitSource = gitSource;
        this.targetDirectory = targetDirectory;
    }
    
    @Override
    public String getDescription() {
        return "Clone Git repository";
    }
    
    @Override
    public int getOrder() {
        return 50;
    }
    
    @Override
    public Satellite getSatellite() {
        // Run locally on the XL Deploy server
        return null;
    }
    
    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        String id = gitSource != null ? gitSource.getId() : "unknown";
        String url = gitSource != null ? (String) gitSource.getProperty("url") : "unknown";
        String branch = gitSource != null ? (String) gitSource.getProperty("branch") : "main";
        
        ctx.logOutput("Starting Git operation for source id: " + id);
        ctx.logOutput("Repository URL: " + url);
        ctx.logOutput("Branch: " + branch);
        
        try {
            // Get Git username from gitSource configuration
            String gitSourceUsername = gitSource != null ? (String) gitSource.getProperty("username") : null;
            String username = gitSourceUsername != null ? gitSourceUsername : "unknown";
            ctx.logOutput("Git Source User: " + username);
            
            // Generate deterministic folder name: Folder_{hash}_username_reponame_branch
            // Hash is based on gitSource ID so each source always uses the same folder
            String workspaceFolderName = YamlHandler.createGitRepoFolderName(id, username, url, branch);
            ctx.logOutput("Git Source ID: " + id);
            
            // Use work/git_repos under server home directory
            String serverHome = System.getProperty("user.dir");
            File gitReposDir = new File(serverHome, "work" + File.separator + "git_repos");
            File localRepoDir = new File(gitReposDir, workspaceFolderName);
            
            ctx.logOutput("Server home: " + serverHome);
            ctx.logOutput("Git repos directory: " + gitReposDir.getAbsolutePath());
            ctx.logOutput("Workspace folder name: " + workspaceFolderName);
            
            String actualTargetDirectory = localRepoDir.getAbsolutePath();
            String operation;
            
            // Check if repository already exists
            if (GitOpsServiceHolder.getGitOpsService().isGitRepository(actualTargetDirectory)) {
                ctx.logOutput("Repository exists - performing pull operation");
                operation = "pull";
            } else {
                ctx.logOutput("Repository doesn't exist - performing clone operation");
                operation = "clone";
            }
            
            ctx.logOutput("Operation: " + operation.toUpperCase());
            ctx.logOutput("Target directory: " + actualTargetDirectory);
            
            String finalClonedDirectory;
            
            if ("clone".equals(operation)) {
                // Perform clone operation
                ctx.logOutput("Starting clone operation...");
                finalClonedDirectory = GitOpsServiceHolder.getGitOpsService().cloneRepository(gitSource, actualTargetDirectory);
                ctx.logOutput("Git repository cloned successfully to: " + finalClonedDirectory);
                
            } else if ("pull".equals(operation)) {
                // Perform pull operation
                ctx.logOutput("Starting pull operation...");
                finalClonedDirectory = GitOpsServiceHolder.getGitOpsService().pullRepository(actualTargetDirectory, 
                    extractCredentials(gitSource));
                ctx.logOutput("Git repository updated successfully at: " + finalClonedDirectory);
                
            } else {
                throw new RuntimeException("Unknown operation: " + operation);
            }
            
            // Log repository information
            logRepositoryInfo(ctx, finalClonedDirectory, operation);
            
            ctx.logOutput("Git " + operation + " operation completed successfully");
            
            return StepExitCode.SUCCESS;
            
        } catch (Exception exception) {
            // Handle exceptions with detailed logging
            String errorMessage = exception.getCause() != null ? 
                exception.getCause().getMessage() : exception.getMessage();
            ctx.logError("Failed to perform Git operation: " + errorMessage, exception);
            
            // Additional error context
            ctx.logError("Source ID: " + id);
            ctx.logError("Repository URL: " + url);
            ctx.logError("Target directory: " + targetDirectory);
            
            return StepExitCode.FAIL;
        }
    }

    
    private GitUserCredentials extractCredentials(ConfigurationItem gitSource) {
        if (gitSource == null) {
            return null;
        }
        
        try {
            String username = (String) gitSource.getProperty("username");
            String password = (String) gitSource.getProperty("password");
            
            if (username != null && !username.trim().isEmpty() && 
                password != null && !password.trim().isEmpty()) {
                return new GitUserCredentials(username, password);
            }
        } catch (Exception e) {
            // Ignore credential extraction errors
        }
        
        return null;
    }
    
    private void logRepositoryInfo(ExecutionContext ctx, String repositoryPath, String operation) {
        try {
            // Extract and log the actual repository folder name
            String actualFolderName = DefaultGitOpsService.extractFolderNameFromPath(repositoryPath);
            ctx.logOutput("Repository folder name: " + actualFolderName);
            
            // Log directory contents for verification
            File repoDir = new File(repositoryPath);
            if (repoDir.exists() && repoDir.isDirectory()) {
                File[] files = repoDir.listFiles();
                if (files != null && files.length > 0) {
                    ctx.logOutput("Repository contains " + files.length + " items");
                } else {
                    ctx.logOutput("Repository appears to be empty");
                }
            }
        } catch (Exception e) {
            ctx.logError("Warning: Could not read repository information: " + e.getMessage());
        }
    }
}