package ai.digital.deploy.gitops.step;

import ai.digital.deploy.gitops.service.GitOpsServiceHolder;
import com.xebialabs.deployit.engine.api.ServiceHolder;
import com.xebialabs.deployit.engine.api.XLDAsCodeService;
import com.xebialabs.deployit.engine.api.dto.XLDAsCodeResult;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

/**
 * Step to export Infrastructure CIs to infrastructure.yaml file.
 * This step uses XLDAsCodeService.generate() to export CIs from the 
 * infrastructureTargetDir and writes them to infrastructure.yaml in the git repository.
 */
public class GitGenerateInfrastructureStep implements Step {

    private static final String INFRASTRUCTURE_YAML = "infrastructure.yaml";
    
    private final String gitDirectoryId;
    private final String infrastructureTargetDir;

    public GitGenerateInfrastructureStep(String gitDirectoryId, String infrastructureTargetDir) {
        this.gitDirectoryId = gitDirectoryId;
        this.infrastructureTargetDir = infrastructureTargetDir;
    }

    @Override
    public String getDescription() {
        return "Export Infrastructure";
    }

    @Override
    public int getOrder() {
        return 20;
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        ctx.logOutput("Starting Infrastructure export");
        ctx.logOutput("Infrastructure target directory: " + infrastructureTargetDir);
        
        Object directoryAttr = ctx.getAttribute(GitOpsExecutionAttributes.TARGET_DIRECTORY);
        if (directoryAttr == null) {
            ctx.logError("Target directory not found in context. Did the repository validation step run?");
            return StepExitCode.FAIL;
        }

        Path directoryPath = Paths.get(directoryAttr.toString());
        
        // Validate directory exists
        if (!Files.exists(directoryPath) || !Files.isDirectory(directoryPath)) {
            ctx.logError("Directory missing or invalid: " + directoryPath);
            return StepExitCode.FAIL;
        }
        
        Path infrastructureYamlPath = directoryPath.resolve(INFRASTRUCTURE_YAML);
        
        ctx.logOutput("Will generate infrastructure.yaml at: " + infrastructureYamlPath);
        
        return generateYamlFile(infrastructureYamlPath, ctx);
    }

    private StepExitCode generateYamlFile(Path yamlFile, ExecutionContext ctx) {
        XLDAsCodeService xldAsCodeService = ServiceHolder.getXLDAsCodeService();
        GitOpsServiceHolder.GitOpsServiceAdapter adapter = null;

        if (xldAsCodeService == null) {
            ctx.logOutput("XLDAsCodeService not available. Falling back to adapter.");
            adapter = GitOpsServiceHolder.getGitOpsService();
        }

        try {
            // Ensure the path starts with "Infrastructure/"
            String fullPath = ensureInfrastructurePath(infrastructureTargetDir);
            ctx.logOutput("Generating YAML from path: " + fullPath);
            
            // Use the generate method to export CIs from the target directory
            XLDAsCodeResult result;
            if (xldAsCodeService != null) {
                result = xldAsCodeService.generate(
                    fullPath,  // path to generate from
                    false,  // globalPermissions
                    false,  // roles
                    false,  // users
                    false,  // includeSecrets
                    false   // includeDefaults
                );
            } else {
                result = adapter.generate(
                    fullPath,  // path to generate from
                    false,  // globalPermissions
                    false,  // roles
                    false,  // users
                    false,  // includeSecrets
                    false   // includeDefaults
                );
            }
            
            if (result == null) {
                ctx.logError("Generate returned null result for path: " + fullPath);
                return StepExitCode.FAIL;
            }
            
            // Check if generation was successful
            if (!result.isSuccess()) {
                ctx.logError("Failed to generate YAML: " + (result.getError() != null ? result.getError() : "Unknown error"));
                return StepExitCode.FAIL;
            }
            
            // Get the generated YAML content
            String yamlContent = result.getContent();
            
            if (yamlContent == null || yamlContent.trim().isEmpty()) {
                ctx.logOutput("No CIs found in " + infrastructureTargetDir + ". Creating empty infrastructure.yaml.");
                yamlContent = "# No infrastructure CIs found in " + infrastructureTargetDir + "\n";
            }
            
            // Write the YAML content to file
            Files.write(yamlFile, yamlContent.getBytes());
            ctx.logOutput("Successfully generated infrastructure.yaml with " + yamlContent.length() + " bytes");
            
            // Log any message from the result
            if (result.getMessage() != null && !result.getMessage().isEmpty()) {
                ctx.logOutput("Generation message: " + result.getMessage());
            }
            
            return StepExitCode.SUCCESS;
            
        } catch (Exception e) {
            ctx.logError("Failed to generate infrastructure.yaml: " + e.getMessage(), e);
            return StepExitCode.FAIL;
        }
    }
    
    /**
     * Ensures the path starts with "Infrastructure/".
     * If the path doesn't start with "Infrastructure/", it will be prepended.
     */
    private String ensureInfrastructurePath(String path) {
        if (path == null || path.trim().isEmpty()) {
            return path;
        }
        String trimmedPath = path.trim();
        if (trimmedPath.startsWith("Infrastructure/") || trimmedPath.equals("Infrastructure")) {
            return trimmedPath;
        }
        return "Infrastructure/" + trimmedPath;
    }
}
