package ai.digital.deploy.gitops.validation;

import com.xebialabs.deployit.engine.api.ServiceHolder;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.Rule;
import com.xebialabs.deployit.plugin.api.validation.ValidationContext;
import com.xebialabs.deployit.plugin.api.validation.Validator;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Validates GitDirectory CI properties when the CI is saved.
 * This annotation should be applied to the git.GitDirectory type in synthetic.xml.
 * 
 * Validates (if provided):
 * - infrastructureTargetDir: must start with "Infrastructure/", be root-level only, and exist
 * - environmentTargetDir: must start with "Environments/", be root-level only, and exist
 * 
 * Note: The "contains CIs" validation is done at export time, not save time,
 * because folders might be empty when saving but populated before export.
 */
@Retention(RetentionPolicy.RUNTIME)
@Rule(clazz = GitDirectoryValidator.GitDirectoryValidatorImpl.class, type = "git-directory-validator")
@Target(ElementType.TYPE)
public @interface GitDirectoryValidator {

    class GitDirectoryValidatorImpl implements Validator<ConfigurationItem> {

        @Override
        public void validate(ConfigurationItem ci, ValidationContext context) {
            String infrastructureTargetDir = ci.getProperty("infrastructureTargetDir");
            String environmentTargetDir = ci.getProperty("environmentTargetDir");

            // Validate infrastructureTargetDir if provided
            if (infrastructureTargetDir != null && !infrastructureTargetDir.trim().isEmpty()) {
                validateTargetDirectory(infrastructureTargetDir.trim(), "Infrastructure", "Infrastructure Target Directory", context);
            }

            // Validate environmentTargetDir if provided
            if (environmentTargetDir != null && !environmentTargetDir.trim().isEmpty()) {
                validateTargetDirectory(environmentTargetDir.trim(), "Environments", "Environment Target Directory", context);
            }
        }

        /**
         * Validates a target directory:
         * 1. Must start with the proper prefix (Infrastructure/ or Environments/)
         * 2. Must be a root-level folder only (exactly one slash)
         * 3. Must exist in Deploy repository
         */
        private void validateTargetDirectory(String targetDir, String rootPrefix, String fieldName, ValidationContext context) {
            // 1. Must start with the proper prefix
            if (!targetDir.startsWith(rootPrefix + "/")) {
                context.error("Invalid %s format: '%s'.The path must start with '%s/' (e.g., '%s/dev').", 
                    fieldName, targetDir, rootPrefix, rootPrefix);
                return;
            }

            // 2. Must have exactly one slash (prefix/folder at root level)
            long slashCount = targetDir.chars().filter(ch -> ch == '/').count();
            if (slashCount != 1) {
                context.error("Invalid %s: '%s'.Only root-level folders can be exported (e.g., '%s/dev').  Subfolders won’t be exported.", 
                    fieldName, targetDir, rootPrefix);
                return;
            }

            // 3. Must exist in Deploy repository
            try {
                boolean exists = ServiceHolder.getRepositoryService().exists(targetDir);
                if (!exists) {
                    context.error("%s '%s' does not exist in Deploy repository. Choose an existing folder or create it first.", fieldName, targetDir);
                }
            } catch (Exception e) {
                context.error("Unable to verify if %s '%s' exists: %s", fieldName, targetDir, e.getMessage());
            }
        }
    }
}
