package ai.digital.deploy.gitops;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.engine.api.ServiceHolder;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Delegate;
import com.xebialabs.deployit.plugin.api.udm.Parameters;
import com.xebialabs.deployit.util.PasswordEncrypter;
import ai.digital.deploy.gitops.step.CheckGitConnectionStep;
import ai.digital.deploy.gitops.step.GitApplyEnvironmentStep;
import ai.digital.deploy.gitops.step.GitApplyInfrastructureStep;
import ai.digital.deploy.gitops.step.GitCommitAndPushStep;
import ai.digital.deploy.gitops.step.GitGenerateEnvironmentStep;
import ai.digital.deploy.gitops.step.GitGenerateInfrastructureStep;
import ai.digital.deploy.gitops.step.GitRepositoryValidationStep;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;

public final class GitOpsDelegate {

    private GitOpsDelegate() {}

    @Delegate(name = "gitOpsCheckConnection")
    public static List<Step> checkConnection(ConfigurationItem item, String name, Map<String, String> args, Parameters params) {
        return Lists.<Step>newArrayList(new CheckGitConnectionStep(item));
    }

    @Delegate(name = "gitOpsImportDirectory")
    public static List<Step> importDirectory(ConfigurationItem gitDirectory, String name, Map<String, String> args, Parameters params) {
        // Get parent GitSource from property or derive from ID
        ConfigurationItem gitSource = gitDirectory.getProperty("gitSource");
        
        // If not set, derive from the directory's ID (e.g., Configuration/testGit/testDir -> Configuration/testGit)
        if (gitSource == null) {
            String directoryId = gitDirectory.getId();
            if (directoryId != null && directoryId.contains("/")) {
                int lastSlash = directoryId.lastIndexOf('/');
                String parentId = directoryId.substring(0, lastSlash);
                // Read parent from repository using engine-api ServiceHolder
                try {
                    gitSource = ServiceHolder.getRepositoryService().read(parentId);
                } catch (Exception e) {
                    throw new RuntimeException("Could not find parent GitSource with ID: " + parentId, e);
                }
            }
        }
        
        String gitRepositoryPath = (String) gitDirectory.getProperty("gitRepositoryPath");
        String gitDirectoryId = gitDirectory.getId();
        String branch = (String) gitDirectory.getProperty("branch");
        
        // Extract and decrypt credentials on the master (delegate runs on master)
        String url = (String) gitSource.getProperty("url");
        String username = (String) gitSource.getProperty("username");
        String encryptedPassword = (String) gitSource.getProperty("password");
        
        String password = null;
        if (encryptedPassword != null && !encryptedPassword.isEmpty()) {
            password = PasswordEncrypter.getInstance().ensureDecrypted(encryptedPassword);
        }
        
        return Lists.<Step>newArrayList(
            new GitRepositoryValidationStep(url, username, password, gitRepositoryPath, gitDirectoryId, branch),
            new GitApplyInfrastructureStep(gitDirectoryId),
            new GitApplyEnvironmentStep(gitDirectoryId)
        );
    }

    @Delegate(name = "gitOpsExportDirectory")
    public static List<Step> exportDirectory(ConfigurationItem gitDirectory, String name, Map<String, String> args, Parameters params) {
        // Get parent GitSource from property or derive from ID
        ConfigurationItem gitSource = gitDirectory.getProperty("gitSource");
        
        // If not set, derive from the directory's ID (e.g., Configuration/testGit/testDir -> Configuration/testGit)
        if (gitSource == null) {
            String directoryId = gitDirectory.getId();
            if (directoryId != null && directoryId.contains("/")) {
                int lastSlash = directoryId.lastIndexOf('/');
                String parentId = directoryId.substring(0, lastSlash);
                // Read parent from repository using engine-api ServiceHolder
                try {
                    gitSource = ServiceHolder.getRepositoryService().read(parentId);
                } catch (Exception e) {
                    throw new RuntimeException("Could not find parent GitSource with ID: " + parentId, e);
                }
            }
        }
        
        String gitRepositoryPath = (String) gitDirectory.getProperty("gitRepositoryPath");
        String gitDirectoryId = gitDirectory.getId();
        String branch = (String) gitDirectory.getProperty("branch");
        String infrastructureSourceDir = (String) gitDirectory.getProperty("infrastructureSourceDir");
        String environmentSourceDir = (String) gitDirectory.getProperty("environmentSourceDir");
        
        // Validate that both source directories are provided (mandatory for export)
        // Note: Root-level and existence validations are done at CI save time by GitDirectoryValidator
        if (infrastructureSourceDir == null || infrastructureSourceDir.trim().isEmpty()) {
            throw new RuntimeException("Infrastructure Source Directory is mandatory for export. Please provide a value (e.g., 'Infrastructure/dev').");
        }
        if (environmentSourceDir == null || environmentSourceDir.trim().isEmpty()) {
            throw new RuntimeException("Environment Source Directory is mandatory for export. Please provide a value (e.g., 'Environments/prod').");
        }
        
        // Validate that source directories contain CIs to export (export-time only check)
        validatePathContainsCIs(infrastructureSourceDir, "Infrastructure Source Directory");
        validatePathContainsCIs(environmentSourceDir, "Environment Source Directory");
        
        // Extract and decrypt credentials on the master (delegate runs on master)
        String url = (String) gitSource.getProperty("url");
        String username = (String) gitSource.getProperty("username");
        String encryptedPassword = (String) gitSource.getProperty("password");
        
        String password = null;
        if (encryptedPassword != null && !encryptedPassword.isEmpty()) {
            password = PasswordEncrypter.getInstance().ensureDecrypted(encryptedPassword);
        }
        
        // Get user-provided tag name from parameters
        String tagName = null;
        String commitMessage = null;
        if (params != null) {
            tagName = params.getProperty("tagName");
            commitMessage = params.getProperty("commitMessage");
        }
        
        // Validate commit message is provided
        if (commitMessage == null || commitMessage.trim().isEmpty()) {
            throw new RuntimeException("Commit message is required. Please provide a commit message for the export.");
        }
        commitMessage = commitMessage.trim();
        
        commitMessage = commitMessage.trim();
        
        // Process tag name (optional - no auto-generation)
        if (tagName != null) {
            tagName = tagName.trim();
            if (tagName.isEmpty()) {
                tagName = null;
            } else {
                // Validate tag name format - Git tag names cannot contain spaces or special characters
                if (!isValidGitTagName(tagName)) {
                    throw new RuntimeException("Invalid tag name: '" + tagName + "'. Git tag names cannot contain spaces or special characters (~^:?*[]\\). Use hyphens or underscores instead.");
                }
                // Validate tag name doesn't already exist (immediate validation before task starts)
                validateTagNotExists(url, username, password, gitRepositoryPath, gitDirectoryId, branch, tagName);
            }
        }
        
        return Lists.<Step>newArrayList(
            new GitRepositoryValidationStep(url, username, password, gitRepositoryPath, gitDirectoryId, branch),
            new GitGenerateInfrastructureStep(gitDirectoryId, infrastructureSourceDir),
            new GitGenerateEnvironmentStep(gitDirectoryId, environmentSourceDir),
            new GitCommitAndPushStep(gitDirectoryId, username, password, tagName, commitMessage, url, branch, gitRepositoryPath, infrastructureSourceDir, environmentSourceDir)
        );
    }
    
    /**
     * Validates that the given tag doesn't already exist in the Git repository.
     * This runs immediately when the user clicks Execute, before any steps are created.
     * Throws RuntimeException if tag already exists, which will show as an error in the UI.
     */
    private static void validateTagNotExists(String url, String username, String password, 
                                              String gitRepositoryPath, String gitDirectoryId, 
                                              String branch, String tagName) {
        try {
            // Use ls-remote to check tags directly from remote - simpler and always current
            UsernamePasswordCredentialsProvider credentials = 
                (username != null && !username.isEmpty() && password != null && !password.isEmpty())
                    ? new UsernamePasswordCredentialsProvider(username, password) 
                    : null;
            
            Collection<Ref> remoteTags = Git.lsRemoteRepository()
                .setRemote(url)
                .setTags(true)
                .setCredentialsProvider(credentials)
                .call();
            
            String tagRefName = "refs/tags/" + tagName;
            for (Ref tagRef : remoteTags) {
                if (tagRef.getName().equals(tagRefName)) {
                    throw new RuntimeException("Tag '" + tagName + "' already exists in the repository. Enter a different tag name, or leave it blank to auto-generate one.");
                }
            }
        } catch (RuntimeException e) {
            throw e;
        } catch (Exception e) {
            // Network issues, etc. - continue, the actual step will fail with more context
        }
    }
    
    /**
     * Validates that the specified path contains Configuration Items (CIs) to export.
     * Throws RuntimeException if the path is empty (contains no CIs).
     * This validation is done at export time only, not at CI save time.
     */
    private static void validatePathContainsCIs(String path, String fieldName) {
        if (path == null || path.trim().isEmpty()) {
            return;
        }
        
        // Use query with parent parameter to find direct children
        List<ConfigurationItemId> children = ServiceHolder.getRepositoryService().query(
            null,           // type - any type
            path.trim(),    // parent - direct children of this path
            null,           // ancestor
            null,           // namePattern
            null,           // lastModifiedBefore
            null,           // lastModifiedAfter
            0,              // page
            1               // resultsPerPage - we only need to know if at least one exists
        );
        if (children == null || children.isEmpty()) {
            throw new RuntimeException(fieldName + " '" + path + "' is empty. No Configuration Items found to export.");
        }
    }
    
    /**
     * Validates that the given tag name is a valid Git tag name.
     * Git tag names cannot contain spaces or special characters: ~ ^ : ? * [ ] \
     * Also cannot start or end with a dot, or contain consecutive dots.
     */
    private static boolean isValidGitTagName(String tagName) {
        if (tagName == null || tagName.isEmpty()) {
            return false;
        }
        // Git tag names cannot contain: ~ ^ : ? * [ ] \ or whitespace
        // Also cannot start/end with dot or contain ..
        String invalidPattern = ".*[~^:?*\\[\\]\\\\\\s].*";
        if (tagName.matches(invalidPattern)) {
            return false;
        }
        if (tagName.startsWith(".") || tagName.endsWith(".") || tagName.contains("..")) {
            return false;
        }
        return true;
    }
}