package ai.digital.deploy.gitops.step;

import ai.digital.deploy.gitops.service.GitOpsServiceHolder;
import ai.digital.deploy.gitops.util.AsCodeResponseLogger;
import ai.digital.deploy.gitops.util.SCMInfo;
import ai.digital.deploy.gitops.util.SCMInfoExtractor;
import ai.digital.deploy.gitops.util.YamlKindFilter;
import ai.digital.deploy.gitops.util.YamlKindFilter.FilterResult;
import com.xebialabs.deployit.engine.api.ServiceHolder;
import com.xebialabs.deployit.engine.api.XLDAsCodeService;
import com.xebialabs.deployit.engine.api.dto.XLDAsCodeResult;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

/**
 * Step to import Environment CIs from environment.yaml file.
 * This step looks specifically for environment.yaml in the target directory.
 * If the file is not found, the step fails and the user must manually skip it.
 */
public class GitApplyEnvironmentStep implements Step {

    private static final String ENVIRONMENT_YAML = "environment.yaml";
    
    private final String gitDirectoryId;

    public GitApplyEnvironmentStep(String gitDirectoryId) {
        this.gitDirectoryId = gitDirectoryId;
    }

    @Override
    public String getDescription() {
        return "Import Environment";
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        ctx.logOutput("Starting Environment import validation and apply");
        
        Object directoryAttr = ctx.getAttribute(GitOpsExecutionAttributes.TARGET_DIRECTORY);
        if (directoryAttr == null) {
            ctx.logError("Target directory not found in context. Did the repository validation step run?");
            return StepExitCode.FAIL;
        }

        Path directoryPath = Paths.get(directoryAttr.toString());
        
        // Validate directory exists
        if (!Files.exists(directoryPath) || !Files.isDirectory(directoryPath)) {
            ctx.logError("Directory missing or invalid: " + directoryPath);
            return StepExitCode.FAIL;
        }
        
        Path environmentYamlPath = directoryPath.resolve(ENVIRONMENT_YAML);
        
        ctx.logOutput("Looking for environment.yaml in: " + directoryPath);

        if (!Files.exists(environmentYamlPath)) {
            ctx.logError("environment.yaml not found in directory: " + directoryPath);
            ctx.logError("Please ensure the file exists or skip this step manually.");
            return StepExitCode.FAIL;
        }

        ctx.logOutput("Found environment.yaml: " + environmentYamlPath);
        
        return applyYamlFile(environmentYamlPath, ctx);
    }

    private StepExitCode applyYamlFile(Path yamlFile, ExecutionContext ctx) {
        XLDAsCodeService xldAsCodeService = ServiceHolder.getXLDAsCodeService();
        GitOpsServiceHolder.GitOpsServiceAdapter adapter = null;

        if (xldAsCodeService == null) {
            ctx.logOutput("XLDAsCodeService not available. Falling back to adapter.");
            adapter = GitOpsServiceHolder.getGitOpsService();
        }

        try {
            String yamlContent = new String(Files.readAllBytes(yamlFile));

            if (yamlContent.trim().isEmpty()) {
                ctx.logOutput("environment.yaml is empty. Skipping.");
                return StepExitCode.SUCCESS;
            }

            // Check for multiple spec.directory entries and warn
            checkMultipleSpecDirectories(yamlContent, ctx);
            
            // Filter YAML content to only include Environments kind
            ctx.logOutput("Validating YAML content for Environments kind...");
            FilterResult filterResult = YamlKindFilter.filterByKind(
                yamlContent, YamlKindFilter.KIND_ENVIRONMENTS, ctx);
            
            // Log any skipped documents
            YamlKindFilter.logSkippedDocuments(filterResult, ENVIRONMENT_YAML, 
                YamlKindFilter.KIND_ENVIRONMENTS, ctx);
            
            ctx.logOutput("  Found " + filterResult.getMatchingDocuments() + " of " + 
                         filterResult.getTotalDocuments() + " document(s) with kind: Environments");
            
            if (!filterResult.hasContent()) {
                ctx.logError("No Environments documents found in environment.yaml.");
                ctx.logError("The file must contain at least one document with 'kind: Environments'.");
                return StepExitCode.FAIL;
            }

            ctx.logOutput("Applying environment.yaml (Environments kind only)...");
            
            // Get SCM traceability data from context and create file-specific copy
            SCMInfo scmData = getSCMDataForFile(ctx, ENVIRONMENT_YAML);
            
            XLDAsCodeResult result;
            if (xldAsCodeService != null) {
                if (scmData != null && scmData.hasCommit()) {
                    ctx.logOutput("Applying with SCM traceability (commit: " + 
                                 scmData.getCommit().substring(0, Math.min(7, scmData.getCommit().length())) + ")");
                    result = xldAsCodeService.applyWithScm(
                        filterResult.getFilteredContent(),
                        scmData.getKind(),
                        scmData.getCommit(),
                        scmData.getAuthor(),
                        scmData.getDate(),
                        scmData.getMessage(),
                        scmData.getRemote(),
                        scmData.getFileName()
                    );
                } else {
                    result = xldAsCodeService.apply(filterResult.getFilteredContent());
                }
            } else {
                // Fallback adapter doesn't support SCM traceability
                result = adapter.apply(filterResult.getFilteredContent());
            }

            if (result.isSuccess()) {
                ctx.logOutput("Successfully applied environment.yaml");
                // Log Created/Updated/Deleted changes like xl CLI
                AsCodeResponseLogger.logChanges(result, ctx);
                return StepExitCode.SUCCESS;
            } else {
                ctx.logError("Failed to apply environment.yaml");;
                ctx.logError("Reason: " + result.getMessage());
                if (result.getError() != null) {
                    ctx.logError("Error: " + result.getError());
                }
                return StepExitCode.FAIL;
            }

        } catch (Exception e) {
            ctx.logError("Exception processing environment.yaml: " + e.getMessage());
            return StepExitCode.FAIL;
        }
    }

    /**
     * Gets SCM traceability data for a specific file from the Git repository.
     * Extracts the commit that last modified this specific file (not HEAD).
     * 
     * @param ctx The execution context
     * @param fileName The name of the YAML file being applied
     * @return SCMInfo with file-specific commit information, or null if not available
     */
    private SCMInfo getSCMDataForFile(ExecutionContext ctx, String fileName) {
        // Get the local repo directory to extract file-specific commit
        Object localRepoDirAttr = ctx.getAttribute(GitOpsExecutionAttributes.LOCAL_REPO_DIR);
        Object targetDirAttr = ctx.getAttribute(GitOpsExecutionAttributes.TARGET_DIRECTORY);
        
        if (localRepoDirAttr != null && targetDirAttr != null) {
            java.io.File localRepoDir = new java.io.File(localRepoDirAttr.toString());
            // Calculate the relative path from repo root to the file
            Path repoPath = localRepoDir.toPath();
            Path targetPath = Paths.get(targetDirAttr.toString());
            Path filePath = targetPath.resolve(fileName);
            
            // Get relative path from repo root
            String relativeFilePath = repoPath.relativize(filePath).toString().replace("\\", "/");
            
            // Extract SCM info for this specific file
            SCMInfo fileScmInfo = SCMInfoExtractor.extractSCMInfo(localRepoDir, relativeFilePath, fileName);
            if (fileScmInfo != null) {
                return fileScmInfo;
            }
        }
        
        // Fallback: Try to use the base SCM info from context
        Object scmAttr = ctx.getAttribute(GitOpsExecutionAttributes.SCM_TRACEABILITY_DATA);
        if (scmAttr instanceof SCMInfo) {
            SCMInfo baseSCM = (SCMInfo) scmAttr;
            return baseSCM.withFileName(fileName);
        }
        
        // Try to create minimal SCM data with just the remote URL if available
        Object remoteUrlAttr = ctx.getAttribute(GitOpsExecutionAttributes.GIT_REMOTE_URL);
        if (remoteUrlAttr != null) {
            return SCMInfoExtractor.createMinimalSCMInfo(remoteUrlAttr.toString(), fileName);
        }
        
        return null;
    }

    /**
     * Checks for multiple spec.directory entries in the YAML content and logs a warning if found.
     * Counts occurrences of '- directory:' pattern which represents list items with directory field.
     */
    private void checkMultipleSpecDirectories(String yamlContent, ExecutionContext ctx) {
        // Count occurrences of list items with 'directory:' field (e.g., "- directory: some/path")
        java.util.regex.Pattern pattern = java.util.regex.Pattern.compile(
            "^\\s*-\\s*directory\\s*:", 
            java.util.regex.Pattern.MULTILINE);
        java.util.regex.Matcher matcher = pattern.matcher(yamlContent);
        
        int count = 0;
        while (matcher.find()) {
            count++;
        }
        
        if (count > 1) {
            ctx.logError("Warning: Found " + count + " spec.directory entries in environment.yaml. " +
                         "Multiple directory specifications detected. Proceeding with import.");
        }
    }

    @Override
    public int getOrder() {
        return 30;
    }
}
