package ai.digital.deploy.gitops.step;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.Status;
import org.eclipse.jgit.api.AddCommand;
import org.eclipse.jgit.api.CommitCommand;
import org.eclipse.jgit.api.PushCommand;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;
import org.eclipse.jgit.transport.PushResult;
import org.eclipse.jgit.transport.RemoteRefUpdate;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Set;

/**
 * Step to commit generated YAML files and push them to the Git repository.
 * This step stages infrastructure.yaml and environment.yaml files,
 * commits them with a descriptive message, and pushes to the remote repository.
 */
public class GitCommitAndPushStep implements Step {

    private static final String INFRASTRUCTURE_YAML = "infrastructure.yaml";
    private static final String ENVIRONMENT_YAML = "environment.yaml";
    
    private final String gitDirectoryId;
    private final String username;
    private final String password;
    private final String tagName;
    private final String commitMessage;
    private final String remoteUrl;
    private final String branch;
    private final String gitRepositoryPath;
    private final String infrastructureSourceDir;
    private final String environmentSourceDir;

    public GitCommitAndPushStep(String gitDirectoryId, String username, String password, String tagName, String commitMessage,
                                String remoteUrl, String branch, String gitRepositoryPath, 
                                String infrastructureSourceDir, String environmentSourceDir) {
        this.gitDirectoryId = gitDirectoryId;
        this.username = username;
        this.password = password;
        this.tagName = tagName;
        this.commitMessage = commitMessage;
        this.remoteUrl = remoteUrl;
        this.branch = branch;
        this.gitRepositoryPath = gitRepositoryPath;
        this.infrastructureSourceDir = infrastructureSourceDir;
        this.environmentSourceDir = environmentSourceDir;
    }

    @Override
    public String getDescription() {
        return "Commit and Push to Git";
    }

    @Override
    public int getOrder() {
        return 40;
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        Object localRepoDirAttr = ctx.getAttribute(GitOpsExecutionAttributes.LOCAL_REPO_DIR);
        if (localRepoDirAttr == null) {
            ctx.logError("Local repository directory not found in context. Did the repository validation step run?");
            return StepExitCode.FAIL;
        }
        
        Object targetDirAttr = ctx.getAttribute(GitOpsExecutionAttributes.TARGET_DIRECTORY);
        if (targetDirAttr == null) {
            ctx.logError("Target directory not found in context. Did the repository validation step run?");
            return StepExitCode.FAIL;
        }

        File localRepoDir = new File(localRepoDirAttr.toString());
        Path targetDirectory = Paths.get(targetDirAttr.toString());
        
        Git git = null;
        try {
            git = Git.open(localRepoDir);
            
            // Calculate relative paths from repo root for staging
            Path repoPath = localRepoDir.toPath();
            Path relativeTarget = repoPath.relativize(targetDirectory);
            
            String infrastructureRelPath = relativeTarget.resolve(INFRASTRUCTURE_YAML).toString().replace('\\', '/');
            String environmentRelPath = relativeTarget.resolve(ENVIRONMENT_YAML).toString().replace('\\', '/');
            
            // Stage the generated files
            AddCommand addCommand = git.add();
            addCommand.addFilepattern(infrastructureRelPath);
            addCommand.addFilepattern(environmentRelPath);
            addCommand.call();
            
            // Check if there are changes to commit
            Status status = git.status().call();
            // Create a mutable set from the unmodifiable sets returned by status
            Set<String> staged = new java.util.HashSet<>(status.getAdded());
            staged.addAll(status.getChanged());
            
            // Collect files changed for summary
            java.util.List<String> filesChanged = new java.util.ArrayList<>();
            
            if (staged.isEmpty() && status.getModified().isEmpty()) {
                ctx.logOutput("No changes detected. Nothing to commit.");
                return StepExitCode.SUCCESS;
            }
            
            for (String file : staged) {
                filesChanged.add(file);
            }
            for (String file : status.getModified()) {
                // Stage modified files as well
                git.add().addFilepattern(file).call();
                filesChanged.add(file);
            }
            
            // Commit with user-provided message
            CommitCommand commitCommand = git.commit();
            commitCommand.setMessage(commitMessage);
            
            // Set author if username is available
            String author = (username != null && !username.isEmpty()) ? username : "xldeploy";
            if (username != null && !username.isEmpty()) {
                commitCommand.setAuthor(username, username + "@xldeploy.local");
                commitCommand.setCommitter(username, username + "@xldeploy.local");
            }
            
            org.eclipse.jgit.revwalk.RevCommit commit = commitCommand.call();
            String commitId = commit.getId().getName();
            
            // Create a tag for this export only if tagName is provided
            boolean tagCreated = false;
            if (tagName != null && !tagName.isEmpty()) {
                String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
                String tagMessage = String.format("XLD Export at %s", timestamp);
                
                git.tag()
                    .setName(tagName)
                    .setMessage(tagMessage)
                    .setAnnotated(true)
                    .call();
                tagCreated = true;
            }
            
            // Push to remote
            PushCommand pushCommand = git.push();
            
            if (username != null && !username.isEmpty() && password != null && !password.isEmpty()) {
                pushCommand.setCredentialsProvider(
                    new UsernamePasswordCredentialsProvider(username, password)
                );
            }
            
            Iterable<PushResult> pushResults = pushCommand.call();
            
            // Check push results
            boolean pushSuccess = true;
            for (PushResult pushResult : pushResults) {
                for (RemoteRefUpdate update : pushResult.getRemoteUpdates()) {
                    RemoteRefUpdate.Status updateStatus = update.getStatus();
                    
                    if (updateStatus != RemoteRefUpdate.Status.OK && 
                        updateStatus != RemoteRefUpdate.Status.UP_TO_DATE) {
                        pushSuccess = false;
                        ctx.logError("Push failed: " + update.getMessage());
                    }
                }
            }
            
            // Push tags separately only if a tag was created
            if (tagCreated) {
                PushCommand pushTagsCommand = git.push().setPushTags();
                
                if (username != null && !username.isEmpty() && password != null && !password.isEmpty()) {
                    pushTagsCommand.setCredentialsProvider(
                        new UsernamePasswordCredentialsProvider(username, password)
                    );
                }
                
                Iterable<PushResult> tagPushResults = pushTagsCommand.call();
                
                for (PushResult pushResult : tagPushResults) {
                    for (RemoteRefUpdate update : pushResult.getRemoteUpdates()) {
                        RemoteRefUpdate.Status updateStatus = update.getStatus();
                        
                        if (updateStatus != RemoteRefUpdate.Status.OK && 
                            updateStatus != RemoteRefUpdate.Status.UP_TO_DATE) {
                            ctx.logError("Tag push failed: " + update.getMessage());
                            pushSuccess = false;
                        }
                    }
                }
            }
            
            if (pushSuccess) {
                // Mask credentials in remote URL for display
                String displayUrl = maskCredentialsInUrl(remoteUrl);
                
                // Print clean summary
                ctx.logOutput("");
                ctx.logOutput("═══════════════════════════════════════════════════════");
                ctx.logOutput("                    EXPORT SUMMARY                      ");
                ctx.logOutput("═══════════════════════════════════════════════════════");
                ctx.logOutput("  Status         : SUCCESS");
                ctx.logOutput("  Pushed by      : " + author);
                ctx.logOutput("───────────────────────────────────────────────────────");
                ctx.logOutput("  GIT DETAILS");
                ctx.logOutput("  Remote         : " + displayUrl);
                ctx.logOutput("  Branch         : " + branch);
                ctx.logOutput("  Repo Path      : " + gitRepositoryPath);
                ctx.logOutput("  Commit ID      : " + commitId);
                ctx.logOutput("───────────────────────────────────────────────────────");
                ctx.logOutput("  EXPORTED FROM XL DEPLOY");
                ctx.logOutput("  Infrastructure : " + infrastructureSourceDir);
                ctx.logOutput("  Environments   : " + environmentSourceDir);
                ctx.logOutput("───────────────────────────────────────────────────────");
                ctx.logOutput("  FILES CHANGED (" + filesChanged.size() + "):");
                for (String file : filesChanged) {
                    ctx.logOutput("    • " + file);
                }
                ctx.logOutput("───────────────────────────────────────────────────────");
                ctx.logOutput("  Commit Message : " + commitMessage);
                if (tagCreated) {
                    ctx.logOutput("  Tag Created    : " + tagName);
                } else {
                    ctx.logOutput("  Tag Created    : (none)");
                }
                ctx.logOutput("═══════════════════════════════════════════════════════");
                ctx.logOutput("");
                return StepExitCode.SUCCESS;
            } else {
                ctx.logError("Push to remote repository failed");
                return StepExitCode.FAIL;
            }
            
        } catch (Exception e) {
            ctx.logError("Git commit and push failed: " + e.getMessage(), e);
            return StepExitCode.FAIL;
        } finally {
            if (git != null) {
                git.close();
            }
        }
    }
    
    /**
     * Masks credentials in a Git URL for safe display in logs.
     * Handles URLs like: https://user:password@github.com/repo.git
     * Returns: https://***@github.com/repo.git (or original URL if no credentials)
     */
    private String maskCredentialsInUrl(String url) {
        if (url == null) {
            return "(unknown)";
        }
        // Pattern to match credentials in URL: protocol://user:pass@host or protocol://user@host
        return url.replaceAll("(https?://)([^:@]+)(:[^@]+)?@", "$1***@");
    }
}
