package ai.digital.deploy.gitops.step;

import ai.digital.deploy.gitops.service.GitOpsServiceHolder;
import com.xebialabs.deployit.engine.api.ServiceHolder;
import com.xebialabs.deployit.engine.api.XLDAsCodeService;
import com.xebialabs.deployit.engine.api.dto.XLDAsCodeResult;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

/**
 * Step to export Environment CIs to environment.yaml file.
 * This step uses XLDAsCodeService.generate() to export CIs from the 
 * environmentSourceDir and writes them to environment.yaml in the git repository.
 */
public class GitGenerateEnvironmentStep implements Step {

    private static final String ENVIRONMENT_YAML = "environment.yaml";
    
    private final String gitDirectoryId;
    private final String environmentSourceDir;

    public GitGenerateEnvironmentStep(String gitDirectoryId, String environmentSourceDir) {
        this.gitDirectoryId = gitDirectoryId;
        this.environmentSourceDir = environmentSourceDir;
    }

    @Override
    public String getDescription() {
        return "Export Environment";
    }

    @Override
    public int getOrder() {
        return 30;
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        ctx.logOutput("Starting Environment export");
        ctx.logOutput("Environment source directory: " + environmentSourceDir);
        
        Object directoryAttr = ctx.getAttribute(GitOpsExecutionAttributes.TARGET_DIRECTORY);
        if (directoryAttr == null) {
            ctx.logError("Target directory not found in context. Did the repository validation step run?");
            return StepExitCode.FAIL;
        }

        Path directoryPath = Paths.get(directoryAttr.toString());
        
        // Validate directory exists
        if (!Files.exists(directoryPath) || !Files.isDirectory(directoryPath)) {
            ctx.logError("Directory missing or invalid: " + directoryPath);
            return StepExitCode.FAIL;
        }
        
        Path environmentYamlPath = directoryPath.resolve(ENVIRONMENT_YAML);
        
        ctx.logOutput("Will generate environment.yaml at: " + environmentYamlPath);
        
        return generateYamlFile(environmentYamlPath, ctx);
    }

    private StepExitCode generateYamlFile(Path yamlFile, ExecutionContext ctx) {
        XLDAsCodeService xldAsCodeService = ServiceHolder.getXLDAsCodeService();
        GitOpsServiceHolder.GitOpsServiceAdapter adapter = null;

        if (xldAsCodeService == null) {
            ctx.logOutput("XLDAsCodeService not available. Falling back to adapter.");
            adapter = GitOpsServiceHolder.getGitOpsService();
        }

        try {
            // Ensure the path starts with "Environments/"
            String fullPath = ensureEnvironmentsPath(environmentSourceDir);
            ctx.logOutput("Generating YAML from path: " + fullPath);
            
            // Use the generate method to export CIs from the target directory
            XLDAsCodeResult result;
            if (xldAsCodeService != null) {
                result = xldAsCodeService.generate(
                    fullPath,  // path to generate from
                    false,  // globalPermissions
                    false,  // roles
                    false,  // users
                    false,  // includeSecrets
                    false   // includeDefaults
                );
            } else {
                result = adapter.generate(
                    fullPath,  // path to generate from
                    false,  // globalPermissions
                    false,  // roles
                    false,  // users
                    false,  // includeSecrets
                    false   // includeDefaults
                );
            }
            
            if (result == null) {
                ctx.logError("Generate returned null result for path: " + fullPath);
                return StepExitCode.FAIL;
            }
            
            // Check if generation was successful
            if (!result.isSuccess()) {
                ctx.logError("Failed to generate YAML: " + (result.getError() != null ? result.getError() : "Unknown error"));
                return StepExitCode.FAIL;
            }
            
            // Get the generated YAML content
            String yamlContent = result.getContent();
            
            if (yamlContent == null || yamlContent.trim().isEmpty()) {
                ctx.logOutput("No CIs found in " + environmentSourceDir + ". Creating empty environment.yaml.");
                yamlContent = "# No environment CIs found in " + environmentSourceDir + "\n";
            }
            
            // Write the YAML content to file
            Files.write(yamlFile, yamlContent.getBytes());
            ctx.logOutput("Successfully generated environment.yaml with " + yamlContent.length() + " bytes");
            
            // Log any message from the result
            if (result.getMessage() != null && !result.getMessage().isEmpty()) {
                ctx.logOutput("Generation message: " + result.getMessage());
            }
            
            return StepExitCode.SUCCESS;
            
        } catch (Exception e) {
            ctx.logError("Failed to generate environment.yaml: " + e.getMessage(), e);
            return StepExitCode.FAIL;
        }
    }
    
    /**
     * Ensures the path starts with "Environments/".
     * If the path doesn't start with "Environments/", it will be prepended.
     */
    private String ensureEnvironmentsPath(String path) {
        if (path == null || path.trim().isEmpty()) {
            return path;
        }
        String trimmedPath = path.trim();
        if (trimmedPath.startsWith("Environments/") || trimmedPath.equals("Environments")) {
            return trimmedPath;
        }
        return "Environments/" + trimmedPath;
    }
}
