package ai.digital.deploy.gitops.util;

import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.LogCommand;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.transport.RemoteConfig;

import java.io.File;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.List;

/**
 * Utility class to extract SCM traceability information from a Git repository.
 * Similar to xl-cli's scm.go functionality for populating SCM data.
 */
public class SCMInfoExtractor {

    private static final String SCM_KIND_GIT = "git";
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ISO_INSTANT;

    /**
     * Extracts SCM traceability data from a Git repository.
     * If filePath is provided, returns the commit that last modified that specific file.
     * If filePath is null or empty, returns the HEAD commit.
     *
     * @param repoDir The local Git repository directory
     * @param filePath The path to the file relative to the repository root (e.g., "dev/infrastructure.yaml"), 
     *                 or null/empty to get HEAD commit
     * @param fileName The display name for the file (e.g., "infrastructure.yaml")
     * @return SCMInfo with commit information, or null if extraction fails
     */
    public static SCMInfo extractSCMInfo(File repoDir, String filePath, String fileName) {
        if (repoDir == null || !repoDir.exists()) {
            return null;
        }

        File gitDir = new File(repoDir, ".git");
        if (!gitDir.exists()) {
            return null;
        }

        try (Git git = Git.open(repoDir)) {
            Repository repository = git.getRepository();
            RevCommit targetCommit = null;
            
            // If filePath is provided, get the commit that last modified that file
            if (filePath != null && !filePath.isEmpty()) {
                LogCommand logCommand = git.log()
                    .setMaxCount(1)
                    .addPath(filePath);
                
                Iterable<RevCommit> commits = logCommand.call();
                for (RevCommit commit : commits) {
                    targetCommit = commit;
                    break;  // We only need the first (most recent) commit
                }
            }
            
            // Fall back to HEAD if no file-specific commit found or no filePath provided
            if (targetCommit == null) {
                ObjectId headId = repository.resolve("HEAD");
                if (headId == null) {
                    return null;
                }
                try (RevWalk revWalk = new RevWalk(repository)) {
                    targetCommit = revWalk.parseCommit(headId);
                }
            }

            // Extract commit information
            String commitHash = targetCommit.getName();
            PersonIdent authorIdent = targetCommit.getAuthorIdent();
            String author = formatAuthor(authorIdent);
            String date = formatDate(authorIdent);
            String message = targetCommit.getShortMessage();
            
            // Get remote URL
            String remoteUrl = getRemoteUrl(git);

            return new SCMInfo(
                SCM_KIND_GIT,
                commitHash,
                author,
                date,
                message,
                remoteUrl,
                fileName
            );

        } catch (Exception e) {
            // Log at debug level, return null on failure
            return null;
        }
    }

    /**
     * Formats the author as "Name <email>" like Git does.
     */
    private static String formatAuthor(PersonIdent author) {
        if (author == null) {
            return "unknown";
        }
        String name = author.getName();
        String email = author.getEmailAddress();
        
        if (email != null && !email.isEmpty()) {
            return name + " <" + email + ">";
        }
        return name;
    }

    /**
     * Formats the commit date as ISO-8601 format.
     */
    private static String formatDate(PersonIdent author) {
        if (author == null) {
            return "";
        }
        Instant instant = author.getWhen().toInstant();
        return DATE_FORMATTER.format(instant);
    }

    /**
     * Gets the remote URL from the "origin" remote configuration.
     */
    private static String getRemoteUrl(Git git) {
        try {
            List<RemoteConfig> remotes = git.remoteList().call();
            for (RemoteConfig remote : remotes) {
                if ("origin".equals(remote.getName())) {
                    if (!remote.getURIs().isEmpty()) {
                        return remote.getURIs().get(0).toString();
                    }
                }
            }
        } catch (Exception e) {
            // Ignore, return empty string
        }
        return "";
    }

    /**
     * Creates a simple SCMInfo with minimal information when Git extraction fails.
     * Uses the provided remote URL if available.
     *
     * @param remoteUrl The Git remote URL (from the GitOps configuration)
     * @param fileName The name of the YAML file being applied
     * @return SCMInfo with available information
     */
    public static SCMInfo createMinimalSCMInfo(String remoteUrl, String fileName) {
        return new SCMInfo(
            SCM_KIND_GIT,
            "",     // commit (unknown)
            "",     // author (unknown)
            "",     // date (unknown)
            "",     // message (unknown)
            remoteUrl != null ? remoteUrl : "",
            fileName
        );
    }
}
