package ai.digital.deploy.gitops.util;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Utility class for filtering YAML content by kind.
 * Handles multi-document YAML files (separated by ---) and extracts only
 * documents matching the expected kind.
 */
public class YamlKindFilter {

    public static final String KIND_INFRASTRUCTURE = "Infrastructure";
    public static final String KIND_ENVIRONMENTS = "Environments";
    
    // Pattern to match the kind field in YAML (handles various whitespace)
    private static final Pattern KIND_PATTERN = Pattern.compile("^kind\\s*:\\s*(.+)$", Pattern.MULTILINE);
    
    /**
     * Result of filtering YAML content by kind.
     */
    public static class FilterResult {
        private final String filteredContent;
        private final int matchingDocuments;
        private final int totalDocuments;
        private final List<String> skippedKinds;
        
        public FilterResult(String filteredContent, int matchingDocuments, int totalDocuments, List<String> skippedKinds) {
            this.filteredContent = filteredContent;
            this.matchingDocuments = matchingDocuments;
            this.totalDocuments = totalDocuments;
            this.skippedKinds = skippedKinds;
        }
        
        public String getFilteredContent() {
            return filteredContent;
        }
        
        public int getMatchingDocuments() {
            return matchingDocuments;
        }
        
        public int getTotalDocuments() {
            return totalDocuments;
        }
        
        public List<String> getSkippedKinds() {
            return skippedKinds;
        }
        
        public boolean hasContent() {
            return filteredContent != null && !filteredContent.trim().isEmpty();
        }
        
        public boolean hasSkippedDocuments() {
            return !skippedKinds.isEmpty();
        }
    }
    
    /**
     * Filters YAML content to only include documents with the expected kind.
     * 
     * @param yamlContent The full YAML content (may contain multiple documents)
     * @param expectedKind The expected kind (e.g., "Infrastructure" or "Environments")
     * @param ctx The execution context for logging (can be null)
     * @return FilterResult containing the filtered content and metadata
     */
    public static FilterResult filterByKind(String yamlContent, String expectedKind, ExecutionContext ctx) {
        if (yamlContent == null || yamlContent.trim().isEmpty()) {
            return new FilterResult("", 0, 0, new ArrayList<>());
        }
        
        // Split by document separator (---)
        String[] documents = yamlContent.split("(?m)^---\\s*$");
        
        List<String> matchingDocuments = new ArrayList<>();
        List<String> skippedKinds = new ArrayList<>();
        int totalDocuments = 0;
        
        for (String document : documents) {
            String trimmedDoc = document.trim();
            if (trimmedDoc.isEmpty()) {
                continue;
            }
            
            totalDocuments++;
            
            // Extract the kind from this document
            String kind = extractKind(trimmedDoc);
            
            if (kind == null) {
                // No kind found - log warning and skip
                if (ctx != null) {
                    ctx.logError("  Warning: Document without 'kind' field found, skipping.");
                }
                skippedKinds.add("(no kind)");
                continue;
            }
            
            if (expectedKind.equalsIgnoreCase(kind)) {
                // Kind matches - include this document
                matchingDocuments.add(trimmedDoc);
            } else {
                // Kind doesn't match - skip with warning
                skippedKinds.add(kind);
            }
        }
        
        // Reconstruct the filtered YAML content
        StringBuilder filteredContent = new StringBuilder();
        for (int i = 0; i < matchingDocuments.size(); i++) {
            if (i > 0) {
                filteredContent.append("\n---\n");
            } else {
                filteredContent.append("---\n");
            }
            filteredContent.append(matchingDocuments.get(i));
        }
        
        return new FilterResult(
            filteredContent.toString(),
            matchingDocuments.size(),
            totalDocuments,
            skippedKinds
        );
    }
    
    /**
     * Extracts the 'kind' value from a YAML document.
     * 
     * @param document A single YAML document
     * @return The kind value, or null if not found
     */
    private static String extractKind(String document) {
        Matcher matcher = KIND_PATTERN.matcher(document);
        if (matcher.find()) {
            return matcher.group(1).trim();
        }
        return null;
    }
    
    /**
     * Logs warnings for skipped documents.
     * 
     * @param result The filter result
     * @param yamlFileName The name of the YAML file being processed
     * @param expectedKind The expected kind
     * @param ctx The execution context for logging
     */
    public static void logSkippedDocuments(FilterResult result, String yamlFileName, 
                                            String expectedKind, ExecutionContext ctx) {
        if (result.hasSkippedDocuments()) {
            ctx.logError("  Warning: Found " + result.getSkippedKinds().size() + 
                         " document(s) with non-matching kind in " + yamlFileName);
            for (String skippedKind : result.getSkippedKinds()) {
                ctx.logError("    - Skipped document with kind: '" + skippedKind + 
                             "' (expected: '" + expectedKind + "')");
            }
        }
    }
}
