package ai.digital.deploy.gitops.validation;

import com.xebialabs.deployit.engine.api.ServiceHolder;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.Rule;
import com.xebialabs.deployit.plugin.api.validation.ValidationContext;
import com.xebialabs.deployit.plugin.api.validation.Validator;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Validates GitDirectory CI properties when the CI is saved.
 * This annotation should be applied to the git.GitDirectory type in synthetic.xml.
 * 
 * Validates (if provided):
 * - infrastructureSourceDir: must start with "Infrastructure/", be root-level only, and exist
 * - environmentSourceDir: must start with "Environments/", be root-level only, and exist
 * 
 * Note: The "contains CIs" validation is done at export time, not save time,
 * because folders might be empty when saving but populated before export.
 */
@Retention(RetentionPolicy.RUNTIME)
@Rule(clazz = GitDirectoryValidator.GitDirectoryValidatorImpl.class, type = "git-directory-validator")
@Target(ElementType.TYPE)
public @interface GitDirectoryValidator {

    class GitDirectoryValidatorImpl implements Validator<ConfigurationItem> {

        @Override
        public void validate(ConfigurationItem ci, ValidationContext context) {
            String infrastructureSourceDir = ci.getProperty("infrastructureSourceDir");
            String environmentSourceDir = ci.getProperty("environmentSourceDir");

            // Validate infrastructureSourceDir if provided
            if (infrastructureSourceDir != null && !infrastructureSourceDir.trim().isEmpty()) {
                validateSourceDirectory(infrastructureSourceDir.trim(), "Infrastructure", "Infrastructure Source Directory", context);
            }

            // Validate environmentSourceDir if provided
            if (environmentSourceDir != null && !environmentSourceDir.trim().isEmpty()) {
                validateSourceDirectory(environmentSourceDir.trim(), "Environments", "Environment Source Directory", context);
            }
        }

        /**
         * Validates a source directory:
         * 1. Must start with the proper prefix (Infrastructure/ or Environments/)
         * 2. Must be a root-level folder only (exactly one slash)
         * 3. Must exist in Deploy repository
         */
        private void validateSourceDirectory(String sourceDir, String rootPrefix, String fieldName, ValidationContext context) {      
            // 1. Must start with the proper prefix
            if (!sourceDir.startsWith(rootPrefix + "/")) {
                context.error("Invalid %s format: '%s'.The path must start with '%s/' (e.g., '%s/dev').",
                    fieldName, sourceDir, rootPrefix, rootPrefix);
            }

            // 2. Must have exactly one slash (prefix/folder at root level)
            long slashCount = sourceDir.chars().filter(ch -> ch == '/').count();
            if (slashCount != 1) {
                context.error("Invalid %s: '%s'.Only root-level folders can be exported (e.g., '%s/dev').  Subfolders won't be exported.",
                    fieldName, sourceDir, rootPrefix);
                return;
            }

            // 3. Must exist in Deploy repository
            try {
                boolean exists = ServiceHolder.getRepositoryService().exists(sourceDir);
                if (!exists) {
                    context.error("%s '%s' does not exist in Deploy repository. Choose an existing folder or create it first.", fieldName, sourceDir);
                }
            } catch (Exception e) {
                context.error("Unable to verify if %s '%s' exists: %s", fieldName, sourceDir, e.getMessage());
            }
        }
    }
}
